<?php
declare(strict_types=1);

namespace DataAlign_Woo;

use WP_CLI; // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedClassFound
use WP_CLI_Command; // phpcs:ignore

if (!defined('ABSPATH')) {
    exit;
}

/**
 * WP-CLI commands for DataAlign plugin to help test real API calls.
 */
class DataAlign_CLI extends WP_CLI_Command
{
    /**
     * Ping the API (uses /api/v1/usage as a sanity check) and print the result.
     *
     * ## EXAMPLES
     *   wp dataalign ping
     */
    public function ping(): void
    {
        $client = new DataAlign_API_Client();
        $res = $client->get_usage();
        if (is_wp_error($res)) {
            WP_CLI::error('Ping failed: ' . $res->get_error_message());
        }
        WP_CLI::success('API reachable. Usage payload:');
        WP_CLI::print_value($res, ['format' => 'json']);
    }

    /**
     * Enrich a single product and optionally apply results.
     *
     * ## OPTIONS
     *  --product_id=<id>   The product post ID.
     *  [--apply]           Apply results to the product (update meta/SEO/tags).
     *
     * ## EXAMPLES
     *   wp dataalign enrich --product_id=123
     *   wp dataalign enrich --product_id=123 --apply
     */
    public function enrich(array $args, array $assoc_args): void
    {
        $product_id = isset($assoc_args['product_id']) ? (int) $assoc_args['product_id'] : 0;
        $apply = isset($assoc_args['apply']);

        if ($product_id <= 0) {
            WP_CLI::error('Invalid or missing --product_id');
        }
        if (!function_exists('wc_get_product')) {
            WP_CLI::error('WooCommerce is required.');
        }
        $product = \wc_get_product($product_id);
        if (!$product) {
            WP_CLI::error('Product not found.');
        }
        $post = \get_post($product_id);
        if (!$post) {
            WP_CLI::error('Product post not found.');
        }

        $title = (string) $post->post_title;
        $description = (string) $post->post_content;
        $categories = \wp_get_post_terms($product_id, 'product_cat', ['fields' => 'names']);
        if (!is_array($categories)) { $categories = []; }

        $images = [];
        $exclude_images = (bool) \get_option('dataalign_exclude_images', false);
        if (!$exclude_images) {
            $thumb_id = (int) \get_post_thumbnail_id($product_id);
            if ($thumb_id) {
                $url = \wp_get_attachment_image_url($thumb_id, 'full');
                if ($url) { $images[] = $url; }
            }
            $gallery_ids = method_exists($product, 'get_gallery_image_ids') ? (array) $product->get_gallery_image_ids() : [];
            foreach ($gallery_ids as $img_id) {
                $url = \wp_get_attachment_image_url((int) $img_id, 'full');
                if ($url) { $images[] = $url; }
            }
            if (method_exists($product, 'is_type') && $product->is_type('variable')) {
                $variation_ids = method_exists($product, 'get_children') ? (array) $product->get_children() : [];
                foreach ($variation_ids as $vid) {
                    $v = \wc_get_product((int) $vid);
                    if ($v && method_exists($v, 'get_image_id')) {
                        $v_img_id = (int) $v->get_image_id();
                        if ($v_img_id) {
                            $u = \wp_get_attachment_image_url($v_img_id, 'full');
                            if ($u) { $images[] = $u; }
                        }
                    }
                }
            }
            $public_base = (string) \get_option('dataalign_public_base_url', '');
            $site_base   = \home_url();
            if ($public_base !== '' && $site_base) {
                $public_base = rtrim($public_base, '/');
                $site_base   = rtrim($site_base, '/');
                $images = array_map(function ($u) use ($public_base, $site_base) { return is_string($u) ? str_replace($site_base, $public_base, $u) : $u; }, $images);
            }
            $images = array_values(array_unique(array_filter(array_map('strval', $images))));
        }

        $client = new DataAlign_API_Client();
        // Gather existing store tags to help API reuse terms
        $existing_tags = [];
        try {
            $terms = \get_terms([
                'taxonomy'   => 'product_tag',
                'hide_empty' => false,
                'fields'     => 'names',
            ]);
            if (!\is_wp_error($terms) && is_array($terms)) {
                $existing_tags = array_values(array_filter(array_map('trim', array_map('strval', $terms))));
            }
        } catch (\Throwable $e) {}
        $payload = [
            'product_id'  => (string) $product_id,
            'title'       => $title,
            'description' => $description,
            'categories'  => array_values(array_map('strval', $categories)),
            'images'      => array_values(array_map('strval', $images)),
            'existing_tags' => $existing_tags,
        ];

        $res = $client->enrich_product($payload);
        if (is_wp_error($res)) {
            WP_CLI::error('Enrich failed: ' . $res->get_error_message());
        }
        WP_CLI::success('Enrich response:');
        WP_CLI::print_value($res, ['format' => 'json']);

        $success = isset($res['success']) ? (bool) $res['success'] : false;
        $data    = isset($res['data']) && is_array($res['data']) ? $res['data'] : [];
        if ($apply && $success && !empty($data)) {
            Plugin::apply_enrichment_data($product_id, $data, $res);
            WP_CLI::success('Applied enrichment data to product #' . $product_id);
        }
    }
}

// Register command when WP-CLI is present.
if (defined('WP_CLI') && WP_CLI) {
    \WP_CLI::add_command('dataalign', DataAlign_CLI::class);
}
