<?php
declare(strict_types=1);
/* phpcs:disable WordPress.WP.I18n.TextDomainMismatch */

namespace DataAlign_Woo;

use WP_Error;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Simple API client for DataAlign API.
 */
class DataAlign_API_Client
{
    /**
     * Register this site and receive an API key.
     * This request is unauthenticated (no API key header).
     *
     * @param array $payload
     * @return array|WP_Error
     */
    public function register_site(array $payload)
    {
        return $this->request_public('POST', '/api/v1/register-site', $payload);
    }
    /**
     * Perform product enrichment for a single product.
     *
     * @param array $payload
     * @return array|WP_Error
     */
    public function enrich_product(array $payload)
    {
        return $this->request('POST', '/api/v1/enrich-product', $payload);
    }

    /**
     * Enrich multiple products in bulk.
     *
     * @param array $products Array of product payloads
     * @return array|WP_Error
     */
    public function enrich_products_bulk(array $products)
    {
        $body = [
            'products' => $products,
        ];
        return $this->request('POST', '/api/v1/enrich-products-bulk', $body);
    }

    /**
     * Get job status by ID.
     *
     * @param int $job_id
     * @return array|WP_Error
     */
    public function get_job(int $job_id)
    {
        $path = '/api/v1/job/' . rawurlencode((string)$job_id);
        return $this->request('GET', $path, null);
    }

    /**
     * Fetch multiple job statuses via GET /api/v1/jobs?ids=1,2,3 (max 50).
     * Response shape: { success: bool, data: { jobs: [{ id, status, result? }] } }
     *
     * @param int[] $job_ids
     * @return array|WP_Error
     */
    public function get_jobs_status(array $job_ids)
    {
        $ids = array_values(array_unique(array_map('intval', $job_ids)));
        if (empty($ids)) {
            return [ 'success' => true, 'data' => [ 'jobs' => [] ] ];
        }
        // Respect backend max 50
        if (count($ids) > 50) { $ids = array_slice($ids, 0, 50); }
        $query = implode(',', array_map('intval', $ids));
        $path = '/api/v1/jobs?ids=' . rawurlencode($query);
        return $this->request('GET', $path, null);
    }

    /**
     * Get current usage info from the API.
     *
     * @return array|WP_Error
     */
    public function get_usage()
    {
        return $this->request('GET', '/api/v1/usage', null);
    }

    /**
     * Core request helper that performs JSON HTTP calls via WP HTTP API.
     *
     * @param string $method HTTP method (GET/POST)
     * @param string $path   API path starting with '/'
     * @param array|null $body  Optional body to JSON-encode
     * @return array|WP_Error
     */
    protected function request(string $method, string $path, ?array $body)
    {
        // Use fixed API base URL
        $base_url = defined(__NAMESPACE__ . '\\API_BASE_URL') ? constant(__NAMESPACE__ . '\\API_BASE_URL') : 'https://api.dataalign.ai';
        $api_key  = (string) get_option('dataalign_api_key', '');
        if ($api_key === '') {
            return new WP_Error('dataalign_missing_api_key', __('DataAlign API key is not configured.', 'ai-product-attributes-for-woocommerce'));
        }

        $url = rtrim($base_url, "/\t\n\r\0\x0B") . (strpos($path, '/') === 0 ? $path : '/' . $path);

        $args = [
            'method'  => strtoupper($method),
            'timeout' => 20,
            'headers' => [
                'Content-Type'    => 'application/json; charset=utf-8',
                'X-DataAlign-Key' => $api_key,
            ],
        ];

        if ($body !== null) {
            $json = wp_json_encode($body);
            if ($json === false) {
                return new WP_Error('dataalign_json_encode_error', __('Failed to encode request body as JSON.', 'ai-product-attributes-for-woocommerce'));
            }
            $args['body'] = $json;
        }

        $response = wp_remote_request($url, $args);
        if (is_wp_error($response)) {
            return $response;
        }

        $code = (int) wp_remote_retrieve_response_code($response);
        $body_str = (string) wp_remote_retrieve_body($response);

        if ($code < 200 || $code >= 300) {
            // Try to decode JSON error to classify plan limit and provide upgrade URL
            $upgrade_url = 'https://dataalign.ai/pricing';
            $err_json = null;
            if ($body_str !== '') {
                $tmp = json_decode($body_str, true);
                if (is_array($tmp)) { $err_json = $tmp; }
            }
            /* translators: %d is the HTTP response status code */
            $message = sprintf(__('Unexpected HTTP status: %d', 'ai-product-attributes-for-woocommerce'), $code);
            if ($body_str !== '') {
                $message .= ' - ' . wp_strip_all_tags($body_str);
            }
            $is_plan_limit = false;
            if (is_array($err_json)) {
                $code_key = isset($err_json['code']) ? (string) $err_json['code'] : '';
                $msg_key  = isset($err_json['message']) ? (string) $err_json['message'] : '';
                if ($code_key !== '') {
                    $is_plan_limit = (strcasecmp($code_key, 'PLAN_LIMIT_REACHED') === 0 || strcasecmp($code_key, 'LIMIT_REACHED') === 0);
                }
                if (!$is_plan_limit && $msg_key !== '') {
                    $m = strtolower($msg_key);
                    if ((strpos($m, 'limit') !== false || strpos($m, 'quota') !== false) && (strpos($m, 'plan') !== false || strpos($m, 'upgrade') !== false)) {
                        $is_plan_limit = true;
                    }
                }
            }
            if (!$is_plan_limit && ($code === 402 || $code === 429)) {
                $is_plan_limit = true;
            }
            if ($is_plan_limit) {
                $friendly = __('You\'ve reached your DataAlign plan limit. Please upgrade your plan to continue.', 'ai-product-attributes-for-woocommerce');
                return new WP_Error('dataalign_plan_limit', $friendly, ['status' => $code, 'upgrade_url' => $upgrade_url, 'api_error' => $err_json]);
            }
            return new WP_Error('dataalign_http_error', $message, ['status' => $code, 'api_error' => $err_json]);
        }

        $decoded = json_decode($body_str, true);
        if (!is_array($decoded)) {
            return new WP_Error('dataalign_json_decode_error', __('Failed to decode response JSON.', 'ai-product-attributes-for-woocommerce'));
        }

        return $decoded;
    }

    /**
     * Public request helper without API key header (for registration flow).
     *
     * @param string $method
     * @param string $path
     * @param array|null $body
     * @return array|WP_Error
     */
    protected function request_public(string $method, string $path, ?array $body)
    {
        // Use fixed API base URL
        $base_url = defined(__NAMESPACE__ . '\\API_BASE_URL') ? constant(__NAMESPACE__ . '\\API_BASE_URL') : 'https://api.dataalign.ai';

        $url = rtrim($base_url, "/\t\n\r\0\x0B") . (strpos($path, '/') === 0 ? $path : '/' . $path);

        $args = [
            'method'  => strtoupper($method),
            'timeout' => 20,
            'headers' => [
                'Content-Type'    => 'application/json; charset=utf-8',
            ],
        ];

        if ($body !== null) {
            $json = wp_json_encode($body);
            if ($json === false) {
                return new WP_Error('dataalign_json_encode_error', __('Failed to encode request body as JSON.', 'ai-product-attributes-for-woocommerce'));
            }
            $args['body'] = $json;
        }

        $response = wp_remote_request($url, $args);
        if (is_wp_error($response)) {
            return $response;
        }

        $code = (int) wp_remote_retrieve_response_code($response);
        $body_str = (string) wp_remote_retrieve_body($response);

        if ($code < 200 || $code >= 300) {
            $upgrade_url = 'https://dataalign.ai/pricing';
            $err_json = null;
            if ($body_str !== '') {
                $tmp = json_decode($body_str, true);
                if (is_array($tmp)) { $err_json = $tmp; }
            }
            /* translators: %d is the HTTP response status code */
            $message = sprintf(__('Unexpected HTTP status: %d', 'ai-product-attributes-for-woocommerce'), $code);
            if ($body_str !== '') {
                $message .= ' - ' . wp_strip_all_tags($body_str);
            }
            $is_plan_limit = false;
            if (is_array($err_json)) {
                $code_key = isset($err_json['code']) ? (string) $err_json['code'] : '';
                $msg_key  = isset($err_json['message']) ? (string) $err_json['message'] : '';
                if ($code_key !== '') {
                    $is_plan_limit = (strcasecmp($code_key, 'PLAN_LIMIT_REACHED') === 0 || strcasecmp($code_key, 'LIMIT_REACHED') === 0);
                }
                if (!$is_plan_limit && $msg_key !== '') {
                    $m = strtolower($msg_key);
                    if ((strpos($m, 'limit') !== false || strpos($m, 'quota') !== false) && (strpos($m, 'plan') !== false || strpos($m, 'upgrade') !== false)) {
                        $is_plan_limit = true;
                    }
                }
            }
            if (!$is_plan_limit && ($code === 402 || $code === 429)) {
                $is_plan_limit = true;
            }
            if ($is_plan_limit) {
                $friendly = __('You\'ve reached your DataAlign plan limit. Please upgrade your plan to continue.', 'ai-product-attributes-for-woocommerce');
                return new WP_Error('dataalign_plan_limit', $friendly, ['status' => $code, 'upgrade_url' => $upgrade_url, 'api_error' => $err_json]);
            }
            return new WP_Error('dataalign_http_error', $message, ['status' => $code, 'api_error' => $err_json]);
        }

        $decoded = json_decode($body_str, true);
        if (!is_array($decoded)) {
            return new WP_Error('dataalign_json_decode_error', __('Failed to decode response JSON.', 'ai-product-attributes-for-woocommerce'));
        }
        return $decoded;
    }
}
