/* global jQuery, DataAlignWooSettings */
(function ($) {
  'use strict';

  function setTestStatus(html) {
    $('#dataalign-test-result').html(html);
  }

  $(document).on('click', '#dataalign-test-connection', function () {
    var $btn = $(this);
    var $spinner = $('#dataalign-settings-spinner');
    $btn.prop('disabled', true);
    $spinner.addClass('is-active');
    setTestStatus('<span>' + (DataAlignWooSettings && DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.testing : 'Testing…') + '</span>');

    $.ajax({
      url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
      method: 'POST',
      dataType: 'json',
      data: {
        action: 'dataalign_test_connection',
        nonce: DataAlignWooSettings && DataAlignWooSettings.nonce
      }
    }).done(function (res) {
      if (res && res.success) {
        setTestStatus('<span style="color:#008a20">' + (DataAlignWooSettings && DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.ok : 'OK') + '</span>');
      } else {
        var emsg = (res && res.data && res.data.message) || (DataAlignWooSettings && DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.failed : 'Failed');
        setTestStatus('<span style="color:#b32d2e">' + emsg + '</span>');
      }
    }).fail(function () {
      var emsg = (DataAlignWooSettings && DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.failed : 'Failed');
      setTestStatus('<span style="color:#b32d2e">' + emsg + '</span>');
    }).always(function () {
      $spinner.removeClass('is-active');
      $btn.prop('disabled', false);
    });
  });

  $(document).on('click', '#dataalign-register-site', function () {
    var $btn = $(this);
    var $spinner = $('#dataalign-settings-spinner');
    $btn.prop('disabled', true);
    $spinner.addClass('is-active');
    setTestStatus('<span>' + (DataAlignWooSettings && DataAlignWooSettings.i18n ? (DataAlignWooSettings.i18n.registering || 'Registering…') : 'Registering…') + '</span>');
    $.ajax({
      url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
      method: 'POST',
      dataType: 'json',
      data: {
        action: 'dataalign_register_site',
        nonce: DataAlignWooSettings && DataAlignWooSettings.nonce
      }
    }).done(function (res) {
      if (res && res.success) {
        setTestStatus('<span style="color:#008a20">' + (DataAlignWooSettings && DataAlignWooSettings.i18n ? (DataAlignWooSettings.i18n.registered || 'Registered') : 'Registered') + '</span>');
        try {
          if (res.data && res.data.api_key) {
            document.getElementById('dataalign_api_key').value = res.data.api_key;
          }
        } catch (e) {}
        // Refresh the page to reflect saved settings and enabled features.
        setTimeout(function () { window.location.reload(); }, 300);
      } else {
        var emsg = (res && res.data && res.data.message) || (DataAlignWooSettings && DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.failed : 'Failed');
        setTestStatus('<span style="color:#b32d2e">' + emsg + '</span>');
      }
    }).fail(function () {
      var emsg = (DataAlignWooSettings && DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.failed : 'Failed');
      setTestStatus('<span style="color:#b32d2e">' + emsg + '</span>');
    }).always(function () {
      $spinner.removeClass('is-active');
      $btn.prop('disabled', false);
    });
  });

  $(document).on('click', '#dataalign-toggle-key', function () {
    var $btn = $(this);
    var input = document.getElementById('dataalign_api_key');
    if (!input) return;
    var current = input.getAttribute('type');
    if (current === 'password') {
      input.setAttribute('type', 'text');
      $btn.text(DataAlignWooSettings && DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.hide : 'Hide');
    } else {
      input.setAttribute('type', 'password');
      $btn.text(DataAlignWooSettings && DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.show : 'Show');
    }
  });

  // Reveal on view for fancy sections
  function reveal() {
    var elems = document.querySelectorAll('.dataalign-reveal');
    var vh = window.innerHeight || document.documentElement.clientHeight;
    for (var i = 0; i < elems.length; i++) {
      var rect = elems[i].getBoundingClientRect();
      if (rect.top < vh - 40) {
        elems[i].classList.add('in-view');
      }
    }
  }
  if (document.readyState !== 'loading') {
    reveal();
  } else {
    document.addEventListener('DOMContentLoaded', reveal);
  }
  window.addEventListener('scroll', reveal, { passive: true });

  // Simple tabs for settings page
  function activateTab(key) {
    var $tabs = $('.dataalign-tabs-nav .nav-tab');
    var $panels = $('.dataalign-tab-panel');
    $tabs.removeClass('nav-tab-active');
    $panels.addClass('hidden');
    var safe = (key || '').replace(/[^a-z0-9_-]/gi, '');
    var $tab = $('.dataalign-tabs-nav .nav-tab[data-tab="' + safe + '"]');
    var $panel = $('#tab-' + safe);
    if ($tab.length && $panel.length) {
      $tab.addClass('nav-tab-active');
      $panel.removeClass('hidden');
      try { window.localStorage.setItem('dataalignSettingsTab', safe); } catch (e) {}
    }
  }

  // Disable other tabs and toggle buttons until API key exists
  function hasApiKey() {
    var input = document.getElementById('dataalign_api_key');
    var val = input ? String(input.value || '').trim() : '';
    if (val) return true;
    return !!(DataAlignWooSettings && DataAlignWooSettings.usageEnabled);
  }

  function updateKeyDependentUI(forceHasKey) {
    var present = (typeof forceHasKey === 'boolean') ? forceHasKey : hasApiKey();
    var $tabs = $('.dataalign-tabs-nav .nav-tab');
    $tabs.each(function () {
      var $t = $(this);
      var key = String($t.data('tab') || '');
      if (key && key !== 'connection') {
        if (present) { $t.removeClass('is-disabled'); }
        else { $t.addClass('is-disabled'); }
      }
    });
    // Toggle buttons
    $('#dataalign-test-connection').toggle(!!present);
    $('#dataalign-register-site').toggle(!present);
    // Force to connection tab if not present
    if (!present) {
      activateTab('connection');
    }
  }

  // Usage loader (used on page load and when starting bulk)
  function loadUsageIfAvailable() {
    try {
      if (!hasApiKey()) return;
      var $usage = $('#dataalign-usage-container');
      if (!$usage.length) return;
      $usage.html('<p>' + (DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.loadingUsage : 'Loading…') + '</p>');
      $.ajax({
        url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
        method: 'POST',
        dataType: 'json',
        data: { action: 'dataalign_get_usage', nonce: DataAlignWooSettings && DataAlignWooSettings.nonce }
      }).done(function (res) {
        if (res && res.success && res.data) {
          var d = res.data;
          var html = '';
          html += '<table class="widefat striped dataalign-usage">';
          html += '<tbody>';
          html += '<tr><td><strong>Month</strong></td><td>' + (d.yearMonth || '') + '</td></tr>';
          html += '<tr><td><strong>Products Processed</strong></td><td>' + (parseInt(d.productsProcessed || 0, 10)) + '</td></tr>';
          html += '<tr><td><strong>Limit</strong></td><td>' + (parseInt(d.limit || 0, 10)) + '</td></tr>';
          html += '<tr><td><strong>Plan</strong></td><td>' + (d.planName || '') + '</td></tr>';
          html += '</tbody></table>';
          $usage.html(html);
        } else {
          var em = (res && res.data && res.data.message) || (DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.usageError : 'Failed');
          $usage.html('<div class="notice notice-error"><p>' + em + '</p></div>');
        }
      }).fail(function () {
        var em = (DataAlignWooSettings.i18n ? DataAlignWooSettings.i18n.usageError : 'Failed');
        $usage.html('<div class="notice notice-error"><p>' + em + '</p></div>');
      });
    } catch (e) {}
  }

  // Background polling for Optimize tab so progress persists across refresh
  var dataalignBulkPollTimer = null;
  function startOptimizePollingIfNeeded() {
    var $panel = $('#tab-optimize');
    if (!$panel.length || $panel.hasClass('hidden')) return;
    if (dataalignBulkPollTimer) return; // already polling

    var $status = $('#dataalign-bulk-enrich-status');
    var $startBtn = $('#dataalign-start-bulk-enrich');
    var $cancelBtn = $('#dataalign-cancel-bulk-enrich');
    var queueBar = document.getElementById('dataalign-bulk-enrich-queue-bar');
    var applyBar = document.getElementById('dataalign-bulk-enrich-apply-bar');
    var queueText = document.getElementById('dataalign-bulk-enrich-queue-text');
    var applyText = document.getElementById('dataalign-bulk-enrich-apply-text');

    function setQueueBar(pct) {
      if (!queueBar) return;
      var clamped = Math.max(0, Math.min(100, pct|0));
      queueBar.style.width = clamped + '%';
      queueBar.setAttribute('aria-valuenow', String(clamped));
    }
    function setApplyBar(pct) {
      if (!applyBar) return;
      var clamped = Math.max(0, Math.min(100, pct|0));
      applyBar.style.width = clamped + '%';
      applyBar.setAttribute('aria-valuenow', String(clamped));
    }

    var lastApplyTick = 0;
    function pollOnce() {
      $.ajax({
        url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
        method: 'POST',
        dataType: 'json',
        // Use poll_now to both advance queue/apply and return latest status
        data: { action: 'dataalign_poll_now', nonce: DataAlignWooSettings && DataAlignWooSettings.nonce }
      }).done(function (res) {
        if (res && res.success && res.data) {
          var s = res.data;
          var total = parseInt(s.total || 0, 10);
          var queued = parseInt(s.queued || s.processed || 0, 10);
          var applied = parseInt(s.applied || 0, 10);
          var remaining = parseInt(s.remaining || Math.max(0, total - applied), 10);
          var status = String(s.status || 'queued');
          var pctQueued = total > 0 ? Math.round((queued / total) * 100) : 0;
          var pctApplied = total > 0 ? Math.round((applied / total) * 100) : 0;
          setQueueBar(pctQueued);
          setApplyBar(pctApplied);
          if (queueText) queueText.textContent = ' ' + queued + '/' + total;
          if (applyText) applyText.textContent = ' ' + applied + '/' + total;
          // Show background note once queuing is complete
          try {
            var note = document.getElementById('dataalign-bulk-background-note');
            if (note) { note.style.display = (total > 0 && queued >= total) ? '' : 'none'; }
          } catch (e) {}
          if ($status.length) {
            if (status === 'completed') {
              $status.html('<span style="color:#008a20">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkDone) || 'Done') + '</span> ' + applied + '/' + total + ' updated.');
            } else if (status === 'cancelled') {
              var cancelledHtml = '';
              if (s && (s.planLimit || s.plan_limit)) {
                var planMsg = (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.planLimitReached) || 'Plan limit reached. Upgrade to continue.';
                cancelledHtml = '<span style="color:#8a6d3b">' + planMsg + '</span>';
                if (s.upgradeUrl) {
                  cancelledHtml += ' — <a class="button button-primary" href="' + String(s.upgradeUrl) + '" target="_blank" rel="noopener noreferrer">Upgrade Plan</a>';
                }
              } else {
                cancelledHtml = '<span style="color:#8a6d3b">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkCancelled) || 'Cancelled') + '</span>';
              }
              $status.html(cancelledHtml);
            } else {
              var label = (status === 'queued') ? (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkQueued) : (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkRunning);
              $status.text((label || 'Running…') + ' ' + applied + '/' + (total || '…') + ' updated — queued: ' + queued + ', remaining: ' + remaining);
            }
          }
          // Update button states based on status so Cancel is enabled during a run
          if ($startBtn.length && $cancelBtn.length) {
            if (status === 'queued' || status === 'running') {
              $startBtn.prop('disabled', true);
              $cancelBtn.prop('disabled', false);
            } else {
              $startBtn.prop('disabled', false);
              $cancelBtn.prop('disabled', true);
            }
          }
          if (status === 'completed' || status === 'cancelled') {
            if (dataalignBulkPollTimer) { clearTimeout(dataalignBulkPollTimer); dataalignBulkPollTimer = null; }
            return;
          }
        }
        dataalignBulkPollTimer = setTimeout(pollOnce, 1500);
      }).fail(function () {
        dataalignBulkPollTimer = setTimeout(pollOnce, 3000);
      });
    }

    pollOnce();
  }

  $(document).on('click', '.dataalign-tabs-nav .nav-tab', function (e) {
    e.preventDefault();
    var $self = $(this);
    if ($self.hasClass('is-disabled')) { return; }
    var key = $self.data('tab');
    if (key) {
      activateTab(String(key));
      if (history && history.replaceState) {
        history.replaceState(null, document.title, window.location.pathname + window.location.search + '#' + key);
      }
      if (String(key) === 'optimize') {
        // If a bulk run is ongoing, start polling to refresh counters and disable Start
        setTimeout(startOptimizePollingIfNeeded, 100);
      }
    }
  });

  $(function () {
    var hash = (window.location.hash || '').replace('#', '');
    var saved = null;
    try { saved = window.localStorage.getItem('dataalignSettingsTab'); } catch (e) {}
    // If no API key yet, always start on connection
    var initial = (hasApiKey() ? (hash || saved || 'connection') : 'connection');
    activateTab(initial);
    updateKeyDependentUI();
    // Load usage immediately when available
    loadUsageIfAvailable();
    if (initial === 'optimize') {
      // If a bulk run is ongoing, start polling to reflect current progress and disable Start
      setTimeout(startOptimizePollingIfNeeded, 150);
    }
  });

  // Watch for manual edits to the API key field to toggle UI
  $(document).on('input change', '#dataalign_api_key', function () {
    updateKeyDependentUI();
  });

  // Restore pre‑AI attributes for all products (background job with progress)
  $(document).on('click', '#dataalign-restore-all-attributes', function () {
    var $btn = $(this);
    var $spinner = $('#dataalign-restore-all-spinner');
    var $status = $('#dataalign-restore-all-status');
    var bar = document.getElementById('dataalign-restore-all-bar');

    function setBar(pct) {
      if (!bar) return;
      var clamped = Math.max(0, Math.min(100, pct|0));
      bar.style.width = clamped + '%';
      bar.setAttribute('aria-valuenow', String(clamped));
    }

    function poll() {
      $.ajax({
        url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
        method: 'POST',
        dataType: 'json',
        data: { action: 'dataalign_get_restore_all_status', nonce: DataAlignWooSettings && DataAlignWooSettings.nonce }
      }).done(function (res) {
        if (res && res.success && res.data) {
          var s = res.data;
          var total = parseInt(s.total || 0, 10);
          var queued = parseInt(s.queued || s.processed || 0, 10);
          var restored = parseInt(s.restored || s.applied || 0, 10);
          var remaining = parseInt(s.remaining || Math.max(0, total - restored), 10);
          var status = String(s.status || 'queued');
          // Progress bar reflects applied/total
          var pct = total > 0 ? Math.round((restored / total) * 100) : 0;
          setBar(pct);
          if (status === 'completed') {
            $status.html('<span style="color:#008a20">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.restoreDone) || 'Restore complete.') + '</span> ' + restored + '/' + total + ' restored.');
            $btn.prop('disabled', false);
            $spinner.removeClass('is-active');
          } else {
            var label = (status === 'queued') ? (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.restoreQueued) : (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.restoreRunning);
            $status.text((label || 'Restoring…') + ' ' + restored + '/' + (total || '…') + ' restored — queued: ' + queued + ', remaining: ' + remaining + ' (running in background)');
            setTimeout(poll, 1500);
          }
        } else {
          var emsg = (res && res.data && res.data.message) || ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
          $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
          $btn.prop('disabled', false);
          $spinner.removeClass('is-active');
        }
      }).fail(function () {
        var emsg = ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
        $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
        $btn.prop('disabled', false);
        $spinner.removeClass('is-active');
      });
    }

    $btn.prop('disabled', true);
    $spinner.addClass('is-active');
    $status.text((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.restoringAll) || 'Restoring products to original state…');
    setBar(0);

    $.ajax({
      url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
      method: 'POST',
      dataType: 'json',
      data: { action: 'dataalign_start_restore_all_attributes', nonce: DataAlignWooSettings && DataAlignWooSettings.nonce }
    }).done(function (res) {
      if (res && res.success) {
        setTimeout(poll, 1200);
      } else {
        var emsg = (res && res.data && res.data.message) || ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
        $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
        $btn.prop('disabled', false);
        $spinner.removeClass('is-active');
      }
    }).fail(function () {
      var emsg = ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
      $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
      $btn.prop('disabled', false);
      $spinner.removeClass('is-active');
    });
  });

  // Bulk optimization (enrich all products)
  $(document).on('click', '#dataalign-start-bulk-enrich', function () {
    var $btn = $(this);
    var $spinner = $('#dataalign-bulk-enrich-spinner');
    var $status = $('#dataalign-bulk-enrich-status');
    var queueBar = document.getElementById('dataalign-bulk-enrich-queue-bar');
    var applyBar = document.getElementById('dataalign-bulk-enrich-apply-bar');
    var queueText = document.getElementById('dataalign-bulk-enrich-queue-text');
    var applyText = document.getElementById('dataalign-bulk-enrich-apply-text');

    // Do not trigger any other side calls on Start; queuing will proceed via polling

    function setQueueBar(pct) {
      if (!queueBar) return;
      var clamped = Math.max(0, Math.min(100, pct|0));
      queueBar.style.width = clamped + '%';
      queueBar.setAttribute('aria-valuenow', String(clamped));
    }
    function setApplyBar(pct) {
      if (!applyBar) return;
      var clamped = Math.max(0, Math.min(100, pct|0));
      applyBar.style.width = clamped + '%';
      applyBar.setAttribute('aria-valuenow', String(clamped));
    }

    function poll() {
      // Use poll_now to both advance queue/apply and return latest status so bars jump in chunks
      $.ajax({
        url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
        method: 'POST',
        dataType: 'json',
        data: { action: 'dataalign_poll_now', nonce: DataAlignWooSettings && DataAlignWooSettings.nonce }
      }).done(function (res) {
        if (res && res.success && res.data) {
          var s = res.data;
          var total = parseInt(s.total || 0, 10);
          var queued = parseInt(s.queued || s.processed || 0, 10);
          var applied = parseInt(s.applied || 0, 10);
          var remaining = parseInt(s.remaining || Math.max(0, total - applied), 10);
          var status = String(s.status || 'queued');
          var pctQueued = total > 0 ? Math.round((queued / total) * 100) : 0;
          var pctApplied = total > 0 ? Math.round((applied / total) * 100) : 0;
          setQueueBar(pctQueued);
          setApplyBar(pctApplied);
          if (queueText) queueText.textContent = ' ' + queued + '/' + total;
          if (applyText) applyText.textContent = ' ' + applied + '/' + total;
          if (status === 'completed') {
            $status.html('<span style="color:#008a20">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkDone) || 'Done') + '</span> ' + applied + '/' + total + ' updated.');
            $btn.prop('disabled', false);
            $spinner.removeClass('is-active');
            if ($cancel.length) $cancel.prop('disabled', true);
          } else if (status === 'cancelled') {
            var cancelledHtml = '';
            if (s && (s.planLimit || s.plan_limit)) {
              var planMsg = (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.planLimitReached) || 'Plan limit reached. Upgrade to continue.';
              cancelledHtml = '<span style="color:#8a6d3b">' + planMsg + '</span>';
              if (s.upgradeUrl) {
                cancelledHtml += ' — <a class="button button-primary" href="' + String(s.upgradeUrl) + '" target="_blank" rel="noopener noreferrer">Upgrade Plan</a>';
              }
            } else {
              cancelledHtml = '<span style="color:#8a6d3b">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkCancelled) || 'Cancelled') + '</span>';
            }
            $status.html(cancelledHtml);
            $btn.prop('disabled', false);
            $spinner.removeClass('is-active');
            if ($cancel.length) $cancel.prop('disabled', true);
          } else {
            var label = (status === 'queued') ? (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkQueued) : (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkRunning);
            $status.text((label || 'Running…') + ' ' + applied + '/' + (total || '…') + ' updated — queued: ' + queued + ', remaining: ' + remaining);
            setTimeout(poll, 1500);
          }
          // Render live logs if present
          try {
            var logs = s.logs || [];
            var box = document.getElementById('dataalign-live-logs');
            if (box && Array.isArray(logs)) {
              box.innerHTML = logs.map(function(line){ return '<div>' + String(line).replace(/</g,'&lt;') + '</div>'; }).join('');
              box.scrollTop = box.scrollHeight;
            }
          } catch (e) {}
        } else {
          var emsg = (res && res.data && res.data.message) || ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
          $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
          $btn.prop('disabled', false);
          $spinner.removeClass('is-active');
        }
      }).fail(function () {
        var emsg = ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
        $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
        $btn.prop('disabled', false);
        $spinner.removeClass('is-active');
      });
    }

    var $cancel = $('#dataalign-cancel-bulk-enrich');
    $btn.prop('disabled', true);
    $spinner.addClass('is-active');
    $status.text((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkStarting) || 'Starting…');
    setQueueBar(0);
    setApplyBar(0);
    if ($cancel.length) $cancel.prop('disabled', false);

    $.ajax({
      url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
      method: 'POST',
      dataType: 'json',
      data: { action: 'dataalign_start_bulk_enrich', nonce: DataAlignWooSettings && DataAlignWooSettings.nonce }
    }).done(function (res) {
      if (res && res.success) {
        setTimeout(poll, 1200);
      } else {
        var emsg = (res && res.data && res.data.message) || ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
        $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
        $btn.prop('disabled', false);
        $spinner.removeClass('is-active');
        if ($cancel.length) $cancel.prop('disabled', true);
      }
    }).fail(function () {
      var emsg = ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
      $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
      $btn.prop('disabled', false);
      $spinner.removeClass('is-active');
      if ($cancel.length) $cancel.prop('disabled', true);
    });
  });

  // Cancel bulk optimization
  $(document).on('click', '#dataalign-cancel-bulk-enrich', function () {
    var $btn = $('#dataalign-start-bulk-enrich');
    var $cancel = $(this);
    var $spinner = $('#dataalign-bulk-enrich-spinner');
    var $status = $('#dataalign-bulk-enrich-status');
    $cancel.prop('disabled', true);
    $spinner.addClass('is-active');
    $status.text((DataAlignWooSettings && DataAlignWooSettings.i18n && (DataAlignWooSettings.i18n.bulkCancelling || 'Cancelling…')) || 'Cancelling…');
    $.ajax({
      url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
      method: 'POST',
      dataType: 'json',
      data: { action: 'dataalign_cancel_bulk_enrich', nonce: DataAlignWooSettings && DataAlignWooSettings.nonce }
    }).done(function (res) {
      $spinner.removeClass('is-active');
      if (res && res.success) {
        $status.html('<span style="color:#8a6d3b">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && (DataAlignWooSettings.i18n.bulkCancelled || 'Cancelled')) || 'Cancelled') + '</span>');
        $btn.prop('disabled', false);
      } else {
        var emsg = (res && res.data && res.data.message) || ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed');
        $status.html('<span style="color:#b32d2e">' + emsg + '</span>');
        $cancel.prop('disabled', false);
      }
    }).fail(function () {
      $spinner.removeClass('is-active');
      $status.html('<span style="color:#b32d2e">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.usageError) || 'Failed') + '</span>');
      $cancel.prop('disabled', false);
    });
  });

  // Poll Now (on-demand apply check)
  $(document).on('click', '#dataalign-poll-now', function () {
    var $btn = $(this);
    var $spinner = $('#dataalign-bulk-enrich-spinner');
    var $status = $('#dataalign-bulk-enrich-status');
    var queueBar = document.getElementById('dataalign-bulk-enrich-queue-bar');
    var applyBar = document.getElementById('dataalign-bulk-enrich-apply-bar');
    var queueText = document.getElementById('dataalign-bulk-enrich-queue-text');
    var applyText = document.getElementById('dataalign-bulk-enrich-apply-text');
    function setQueueBar(pct) { if (!queueBar) return; var p = Math.max(0, Math.min(100, pct|0)); queueBar.style.width = p + '%'; queueBar.setAttribute('aria-valuenow', String(p)); }
    function setApplyBar(pct) { if (!applyBar) return; var p = Math.max(0, Math.min(100, pct|0)); applyBar.style.width = p + '%'; applyBar.setAttribute('aria-valuenow', String(p)); }
    $btn.prop('disabled', true);
    $spinner.addClass('is-active');
    $.ajax({
      url: (DataAlignWooSettings && DataAlignWooSettings.ajaxUrl) || ajaxurl,
      method: 'POST',
      dataType: 'json',
      data: { action: 'dataalign_poll_now', nonce: DataAlignWooSettings && DataAlignWooSettings.nonce }
    }).done(function (res) {
      if (res && res.success && res.data) {
        var s = res.data;
        var total = parseInt(s.total || 0, 10);
        var queued = parseInt(s.queued || s.processed || 0, 10);
        var applied = parseInt(s.applied || 0, 10);
        var remaining = parseInt(s.remaining || Math.max(0, total - applied), 10);
        var status = String(s.status || 'queued');
        var pctQueued = total > 0 ? Math.round((queued / total) * 100) : 0;
        var pctApplied = total > 0 ? Math.round((applied / total) * 100) : 0;
        setQueueBar(pctQueued);
        setApplyBar(pctApplied);
        if (queueText) queueText.textContent = ' ' + queued + '/' + total;
        if (applyText) applyText.textContent = ' ' + applied + '/' + total;
        if (status === 'completed') {
          $status.html('<span style="color:#008a20">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkDone) || 'Done') + '</span> ' + applied + '/' + total + ' updated.');
        } else if (status === 'cancelled') {
          var cancelledHtml = '';
          if (s && (s.planLimit || s.plan_limit)) {
            var planMsg = (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.planLimitReached) || 'Plan limit reached. Upgrade to continue.';
            cancelledHtml = '<span style="color:#8a6d3b">' + planMsg + '</span>';
            if (s.upgradeUrl) {
              cancelledHtml += ' — <a class="button button-primary" href="' + String(s.upgradeUrl) + '" target="_blank" rel="noopener noreferrer">Upgrade Plan</a>';
            }
          } else {
            cancelledHtml = '<span style="color:#8a6d3b">' + ((DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkCancelled) || 'Cancelled') + '</span>';
          }
          $status.html(cancelledHtml);
        } else {
          var label = (status === 'queued') ? (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkQueued) : (DataAlignWooSettings && DataAlignWooSettings.i18n && DataAlignWooSettings.i18n.bulkRunning);
          $status.text((label || 'Running…') + ' ' + applied + '/' + (total || '…') + ' updated — queued: ' + queued + ', remaining: ' + remaining);
        }
      }
    }).always(function () {
      $spinner.removeClass('is-active');
      $btn.prop('disabled', false);
    });
  });
})(jQuery);
