<?php
declare(strict_types=1);
/* phpcs:disable WordPress.WP.I18n.TextDomainMismatch */

/**
 * Plugin Name: AI Product Attributes – Specs, Tags & SEO for WooCommerce
 * Plugin URI: https://dataalign.ai/
 * Description: AI-powered product data enrichment & alignment for WooCommerce, integrating with the DataAlign API.
 * Version: 0.1.0
 * Author: DataAlign.ai
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.6
 * Requires PHP: 7.4
 * Text Domain: ai-product-attributes-specs-tags-seo-for-woocommerce
 * Domain Path: /languages
 */

namespace DataAlign_Woo;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define simple constants
const VERSION = '0.1.0';
const PLUGIN_FILE = __FILE__;
// Fixed API base URL
const API_BASE_URL = 'https://api.dataalign.ai';
// Upgrade/Plans URL
const UPGRADE_URL = 'https://dataalign.ai/pricing';

// Includes
require_once __DIR__ . '/includes/class-dataalign-api-client.php';
// Register WP-CLI commands if available
if (defined('WP_CLI') && WP_CLI) {
    require_once __DIR__ . '/includes/class-dataalign-cli.php';
}

/**
 * Main plugin bootstrap.
 */
final class Plugin
{
    /** Jobs table name */
    protected static function jobs_table(): string
    {
        global $wpdb; return $wpdb->prefix . 'dataalign_jobs';
    }

    /**
     * Jobs table for direct SQL: sanitized and backticked identifier.
     * Use this only when building raw SQL strings; for `$wpdb->update/insert` pass the plain table name.
     */
    protected static function jobs_table_sql(): string
    {
        $table = self::jobs_table();
        // Whitelist safe identifier characters to satisfy security scanners.
        $table = preg_replace('/[^A-Za-z0-9_]/', '', (string) $table);
        return '`' . $table . '`';
    }

    /** Append a log line to a rolling buffer for Optimize UI */
    protected static function log(string $msg): void
    {
        $buf = get_option('_dataalign_logs', []);
        if (!is_array($buf)) { $buf = []; }
        $buf[] = gmdate('H:i:s') . ' ' . $msg;
        if (count($buf) > 300) { $buf = array_slice($buf, -300); }
        update_option('_dataalign_logs', $buf, false);
    }
    /**
     * Resolve a product_tag term ID for a given tag name, reusing existing terms whenever possible.
     * Tries slug match, exact name match, and common-normalized variants (e.g., '&' vs 'and').
     * Returns 0 on failure.
     */
    protected static function resolve_product_tag_id(string $raw_name): int
    {
        static $cache = [];
        $name = trim((string) $raw_name);
        if ($name === '') { return 0; }

        // Basic de-duplication across this request
        $cache_key = strtolower($name);
        if (isset($cache[$cache_key])) {
            return (int) $cache[$cache_key];
        }

        // Build candidate name variants for matching
        $name_space = preg_replace('/\s+/u', ' ', $name);
        $name_amp   = preg_replace('/\s*&\s*/iu', ' and ', $name_space);
        $name_and   = preg_replace('/\s+and\s+/iu', ' & ', $name_space);
        $name_clean = preg_replace("/[\"'\\x{2018}\\x{2019}\\x{201C}\\x{201D}.,:;!\\(\\)\\[\\]\\{\\}\\/\\\\]+/u", '', $name_amp);

        $name_candidates = array_values(array_unique(array_filter([
            $name_space,
            $name_amp,
            $name_and,
            $name_clean,
        ], function ($s) { return (string) $s !== ''; })));

        // Try slug matches for each candidate first (fast and robust)
        $slug_candidates = [];
        foreach ($name_candidates as $cand) {
            $slug = sanitize_title($cand);
            if ($slug !== '') { $slug_candidates[$slug] = true; }
        }
        // Primary slug: based on original name
        $primary_slug = sanitize_title($name);
        if ($primary_slug !== '') { $slug_candidates = [$primary_slug => true] + $slug_candidates; }

        foreach (array_keys($slug_candidates) as $slug) {
            $term = \get_term_by('slug', $slug, 'product_tag');
            if ($term && !\is_wp_error($term) && isset($term->term_id)) {
                $cache[$cache_key] = (int) $term->term_id;
                return (int) $term->term_id;
            }
        }

        // Try exact name matches for each candidate
        foreach ($name_candidates as $cand) {
            $term = \get_term_by('name', $cand, 'product_tag');
            if ($term && !\is_wp_error($term) && isset($term->term_id)) {
                $cache[$cache_key] = (int) $term->term_id;
                return (int) $term->term_id;
            }
        }

        // Finally, create the term using the primary slug. Handle race/exists errors gracefully.
        $created = \wp_insert_term($name_space, 'product_tag', ['slug' => $primary_slug]);
        if (!\is_wp_error($created) && isset($created['term_id'])) {
            $cache[$cache_key] = (int) $created['term_id'];
            return (int) $created['term_id'];
        }
        if (\is_wp_error($created)) {
            $exists_id = $created->get_error_data('term_exists');
            if ($exists_id) {
                $cache[$cache_key] = (int) $exists_id;
                return (int) $exists_id;
            }
            // Last-chance fetch by slug in case another process just created it
            if ($primary_slug !== '') {
                $term2 = \get_term_by('slug', $primary_slug, 'product_tag');
                if ($term2 && !\is_wp_error($term2) && isset($term2->term_id)) {
                    $cache[$cache_key] = (int) $term2->term_id;
                    return (int) $term2->term_id;
                }
            }
        }

        return 0;
    }

    /**
     * Get all existing WooCommerce product tag names in the store.
     * Used to inform the API of reusable tags to avoid constant creation of new tags.
     *
     * @return string[]
     */
    protected static function get_all_product_tag_names(): array
    {
        try {
            $terms = \get_terms([
                'taxonomy'   => 'product_tag',
                'hide_empty' => false,
                'fields'     => 'names',
            ]);
            if (\is_wp_error($terms) || !is_array($terms)) {
                return [];
            }
            $names = array_values(array_filter(array_map('trim', array_map('strval', $terms))));
            return $names;
        } catch (\Throwable $e) {
            return [];
        }
    }
    /** Load translations (handled automatically on WordPress.org; kept for non.org installs) */
    public static function load_textdomain(): void
    {
        // Intentionally left blank to satisfy Plugin Check guidance (WP.org auto-loads translations).
    }
    /**
     * Initialize hooks for admin UI, AJAX, bulk actions, and cron.
     */
    public static function init(): void
    {
        // On WordPress.org, translations load automatically by slug.
        // add_action('plugins_loaded', [__CLASS__, 'load_textdomain']);
        \add_action('admin_menu', [__CLASS__, 'register_admin_menu']);
        \add_action('admin_init', [__CLASS__, 'register_settings']);
        \add_action('admin_notices', [__CLASS__, 'admin_notices']);
        \add_action('admin_notices', [__CLASS__, 'product_requirements_notices']);
        \add_filter('plugin_action_links_' . plugin_basename(PLUGIN_FILE), [__CLASS__, 'plugin_action_links']);
        \add_filter('plugin_row_meta', [__CLASS__, 'plugin_row_meta'], 10, 2);
        // Phase 2 hooks
        \add_action('add_meta_boxes', [__CLASS__, 'register_product_metabox']);
        \add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin_assets']);
        \add_action('wp_ajax_dataalign_enrich_product', [__CLASS__, 'ajax_enrich_product']);
        \add_action('wp_ajax_dataalign_test_connection', [__CLASS__, 'ajax_test_connection']);
        \add_action('wp_ajax_dataalign_register_site', [__CLASS__, 'ajax_register_site']);
        \add_action('wp_ajax_dataalign_get_usage', [__CLASS__, 'ajax_get_usage']);
        \add_action('wp_ajax_dataalign_sync_attributes', [__CLASS__, 'ajax_sync_attributes']);
        \add_action('wp_ajax_dataalign_restore_baseline', [__CLASS__, 'ajax_restore_baseline']);
        // Legacy batch AJAX (kept for compatibility)
        \add_action('wp_ajax_dataalign_restore_all_attributes', [__CLASS__, 'ajax_restore_all_attributes']);
        // Background job controls
        \add_action('wp_ajax_dataalign_start_restore_all_attributes', [__CLASS__, 'ajax_start_restore_all_attributes']);
        \add_action('wp_ajax_dataalign_get_restore_all_status', [__CLASS__, 'ajax_get_restore_all_status']);
        // Background processors (Action Scheduler or WP-Cron fallback)
        \add_action('dataalign_restore_all_attributes_tick', [__CLASS__, 'process_restore_all_tick']);
        \add_action('wp_ajax_dataalign_get_attributes', [__CLASS__, 'ajax_get_attributes']);
        \add_action('wp_ajax_dataalign_get_wc_attributes', [__CLASS__, 'ajax_get_wc_attributes']);
        \add_action('wp_ajax_dataalign_get_product_tags', [__CLASS__, 'ajax_get_product_tags']);
        // Bulk optimization (all products)
        \add_action('wp_ajax_dataalign_start_bulk_enrich', [__CLASS__, 'ajax_start_bulk_enrich']);
        \add_action('wp_ajax_dataalign_get_bulk_enrich_status', [__CLASS__, 'ajax_get_bulk_enrich_status']);
        // Legacy bulk queue tick removed in favor of table-driven pipeline
        \add_action('wp_ajax_dataalign_cancel_bulk_enrich', [__CLASS__, 'ajax_cancel_bulk_enrich']);
        \add_action('wp_ajax_dataalign_poll_now', [__CLASS__, 'ajax_poll_now']);
        // Phase 3 hooks: bulk and cron
        \add_filter('bulk_actions-edit-product', [__CLASS__, 'register_bulk_actions']);
        \add_filter('handle_bulk_actions-edit-product', [__CLASS__, 'handle_bulk_actions'], 10, 3);
        \add_filter('cron_schedules', [__CLASS__, 'register_cron_schedules']);
        \add_action('dataalign_cron_check_jobs', [__CLASS__, 'cron_check_jobs']);
        // Frontend specs tab and shortcode
        \add_filter('woocommerce_product_tabs', [__CLASS__, 'add_specs_tab']);
        // Ensure Additional information tab can be hidden even if other filters add it later
        \add_filter('woocommerce_product_tabs', [__CLASS__, 'maybe_hide_additional_info_tab'], 99);
        \add_shortcode('dataalign_specs', [__CLASS__, 'shortcode_specs']);
        // Frontend: humanize labels in Additional Information tab
        \add_filter('woocommerce_attribute_label', [__CLASS__, 'filter_attribute_label'], 10, 3);
        // Frontend: enqueue styles for product page buttons
        \add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_frontend_assets']);
    }

    /**
     * Detect the active SEO plugin for this site/post and return slug+name.
     * If multiple are active, prefer the one with existing post meta; fallback to precedence.
     */
    protected static function detect_seo_plugin(int $post_id = 0): array
    {
        $candidates = [
            'yoast'     => [
                'active' => (defined('WPSEO_VERSION') || class_exists('WPSEO_Meta')),
                'name'   => 'Yoast SEO',
                'has'    => function ($pid) { return (string) \get_post_meta($pid, '_yoast_wpseo_title', true) !== '' || (string) \get_post_meta($pid, '_yoast_wpseo_metadesc', true) !== ''; },
            ],
            'rank_math' => [
                'active' => (defined('RANK_MATH_VERSION') || class_exists('RankMath\\Helper') || function_exists('rank_math')),
                'name'   => 'Rank Math',
                'has'    => function ($pid) { return (string) \get_post_meta($pid, 'rank_math_title', true) !== '' || (string) \get_post_meta($pid, 'rank_math_description', true) !== ''; },
            ],
            'aioseo'    => [
                'active' => (defined('AIOSEO_VERSION') || class_exists('AIOSEO\\Plugin')),
                'name'   => 'All in One SEO',
                'has'    => function ($pid) { $a = \get_post_meta($pid, '_aioseo', true); return (is_array($a) && ((!empty($a['title'])) || (!empty($a['description'])))) || (string) \get_post_meta($pid, '_aioseop_title', true) !== '' || (string) \get_post_meta($pid, '_aioseop_description', true) !== ''; },
            ],
            'seopress'  => [
                'active' => (defined('SEOPRESS_VERSION') || function_exists('seopress_init')),
                'name'   => 'SEOPress',
                'has'    => function ($pid) { return (string) \get_post_meta($pid, '_seopress_titles_title', true) !== '' || (string) \get_post_meta($pid, '_seopress_titles_desc', true) !== ''; },
            ],
            'tsf'       => [
                'active' => (defined('THE_SEO_FRAMEWORK_VERSION') || class_exists('The_SEO_Framework')),
                'name'   => 'The SEO Framework',
                'has'    => function ($pid) { return (string) \get_post_meta($pid, '_genesis_title', true) !== '' || (string) \get_post_meta($pid, '_genesis_description', true) !== ''; },
            ],
        ];

        $post_id = (int) $post_id;
        $active = [];
        foreach ($candidates as $slug => $info) {
            if ($info['active']) {
                $active[$slug] = $info;
            }
        }
        if (empty($active)) {
            $detected = ['slug' => 'none', 'name' => 'None'];
            return \apply_filters('dataalign_detected_seo_plugin', $detected, $post_id);
        }

        // Prefer the one with existing per-post meta
        if ($post_id > 0) {
            foreach ($active as $slug => $info) {
                $has = false;
                try { $has = (bool) $info['has']($post_id); } catch (\Throwable $e) { $has = false; }
                if ($has) {
                    $detected = ['slug' => $slug, 'name' => (string) $info['name']];
                    return \apply_filters('dataalign_detected_seo_plugin', $detected, $post_id);
                }
            }
        }

        // Fallback precedence order
        $precedence = ['yoast','rank_math','aioseo','seopress','tsf'];
        foreach ($precedence as $slug) {
            if (isset($active[$slug])) {
                $detected = ['slug' => $slug, 'name' => (string) $active[$slug]['name']];
                return \apply_filters('dataalign_detected_seo_plugin', $detected, $post_id);
            }
        }

        $first = array_key_first($active);
        $detected = ['slug' => (string) $first, 'name' => (string) $active[$first]['name']];
        return \apply_filters('dataalign_detected_seo_plugin', $detected, $post_id);
    }

    /**
     * Return character limits for SEO title/description per plugin slug.
     */
    protected static function get_seo_limits(string $slug): array
    {
        $slug = strtolower($slug);
        $limits = [
            'yoast'     => ['title_max' => 60, 'description_max' => 155],
            'rank_math' => ['title_max' => 60, 'description_max' => 160],
            'seopress'  => ['title_max' => 60, 'description_max' => 160],
            'aioseo'    => ['title_max' => 60, 'description_max' => 160],
            'tsf'       => ['title_max' => 60, 'description_max' => 160],
            'default'   => ['title_max' => 60, 'description_max' => 160],
        ];
        $out = $limits[$slug] ?? $limits['default'];
        return \apply_filters('dataalign_seo_plugin_limits', $out, $slug);
    }

    /**
     * Read current SEO title and description from detected SEO plugin for a post.
     */
    protected static function get_current_seo(int $post_id): array
    {
        $post_id = (int) $post_id;
        $out = ['title' => '', 'meta_description' => ''];
        $det = self::detect_seo_plugin($post_id);
        $slug = (string) ($det['slug'] ?? 'none');

        try {
            switch ($slug) {
                case 'yoast':
                    $out['title'] = (string) \get_post_meta($post_id, '_yoast_wpseo_title', true);
                    $out['meta_description'] = (string) \get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
                    break;
                case 'rank_math':
                    $out['title'] = (string) \get_post_meta($post_id, 'rank_math_title', true);
                    $out['meta_description'] = (string) \get_post_meta($post_id, 'rank_math_description', true);
                    break;
                case 'aioseo':
                    $a = \get_post_meta($post_id, '_aioseo', true);
                    if (is_array($a)) {
                        $out['title'] = (string) ($a['title'] ?? '');
                        $out['meta_description'] = (string) ($a['description'] ?? '');
                    }
                    if ($out['title'] === '') { $out['title'] = (string) \get_post_meta($post_id, '_aioseop_title', true); }
                    if ($out['meta_description'] === '') { $out['meta_description'] = (string) \get_post_meta($post_id, '_aioseop_description', true); }
                    break;
                case 'seopress':
                    $out['title'] = (string) \get_post_meta($post_id, '_seopress_titles_title', true);
                    $out['meta_description'] = (string) \get_post_meta($post_id, '_seopress_titles_desc', true);
                    break;
                case 'tsf':
                    $out['title'] = (string) \get_post_meta($post_id, '_genesis_title', true);
                    $out['meta_description'] = (string) \get_post_meta($post_id, '_genesis_description', true);
                    break;
                default:
                    // none
                    break;
            }
        } catch (\Throwable $e) {
            // Ignore and return empty
        }

        $out['title'] = trim($out['title']);
        $out['meta_description'] = trim($out['meta_description']);
        return $out;
    }

    /**
     * Truncate text to a char limit (best-effort, adds ellipsis when trimmed).
     */
    protected static function truncate_to_limit(string $text, int $limit): string
    {
        $text = trim($text);
        if ($limit <= 0 || $text === '') { return $text; }
        // Prefer multibyte if available
        if (function_exists('mb_strlen') && function_exists('mb_substr')) {
            if (mb_strlen($text) <= $limit) { return $text; }
            $truncated = mb_substr($text, 0, max(0, $limit - 1));
            return rtrim($truncated) . '…';
        }
        if (strlen($text) <= $limit) { return $text; }
        $truncated = substr($text, 0, max(0, $limit - 1));
        return rtrim($truncated) . '…';
    }

    /**
     * Add Settings link on the Plugins list page for this plugin.
     *
     * @param array $links
     * @return array
     */
    public static function plugin_action_links(array $links): array
    {
        $url = \admin_url('admin.php?page=dataalign-woo');
        $settings_link = '<a href="' . \esc_url($url) . '">' . \esc_html__('Settings', 'ai-product-attributes-for-woocommerce') . '</a>';
        $website_link  = '<a href="' . \esc_url('https://dataalign.ai/') . '" target="_blank" rel="noopener noreferrer">' . \esc_html__('Website', 'ai-product-attributes-for-woocommerce') . '</a>';
        array_unshift($links, $settings_link);
        $links[] = $website_link;
        return $links;
    }

    /**
     * Add website link to the plugin row meta for marketing/best practices.
     */
    public static function plugin_row_meta(array $links, string $file): array
    {
        if ($file === plugin_basename(PLUGIN_FILE)) {
            $links[] = '<a href="' . \esc_url('https://dataalign.ai/') . '" target="_blank" rel="noopener noreferrer">' . \esc_html__('DataAlign.ai', 'ai-product-attributes-for-woocommerce') . '</a>';
        }
        return $links;
    }

    /**
     * Register submenu under WooCommerce for DataAlign AI settings.
     */
    public static function register_admin_menu(): void
    {
        \add_submenu_page(
            'woocommerce',
            \esc_html__('DataAlign AI', 'ai-product-attributes-for-woocommerce'),
            \esc_html__('DataAlign AI', 'ai-product-attributes-for-woocommerce'),
            'manage_woocommerce',
            'dataalign-woo',
            [__CLASS__, 'render_settings_page']
        );
    }

    /**
     * Register settings, sections, and fields for the settings page.
     */
    public static function register_settings(): void
    {
        // Settings group: dataalign_woo_settings

        \register_setting(
            'dataalign_woo_settings',
            'dataalign_api_key',
            [
                'type' => 'string',
                'sanitize_callback' => function ($value) {
                    $value = is_string($value) ? trim($value) : '';
                    return \sanitize_text_field($value);
                },
                'default' => '',
            ]
        );

        // Optional: Public base URL for media rewrite (e.g., ngrok/Cloudflare tunnel)
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_public_base_url',
            [
                'type' => 'string',
                'sanitize_callback' => function ($value) {
                    $value = is_string($value) ? trim($value) : '';
                    return \esc_url_raw($value, ['http', 'https']);
                },
                'default' => '',
            ]
        );

        // Optional: Exclude images from payload when enriching
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_exclude_images',
            [
                'type' => 'boolean',
                'sanitize_callback' => function ($value) {
                    return (bool) $value;
                },
                'default' => true,
            ]
        );

        // Attribute sync settings
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_sync_attributes_to_woo',
            [
                'type' => 'boolean',
                'sanitize_callback' => function ($value) { return (bool) $value; },
                'default' => false,
            ]
        );
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_enable_specs_tab',
            [
                'type' => 'boolean',
                'sanitize_callback' => function ($value) { return (bool) $value; },
                'default' => false,
            ]
        );
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_hide_additional_info_tab',
            [
                'type' => 'boolean',
                'sanitize_callback' => function ($value) { return (bool) $value; },
                'default' => false,
            ]
        );
        // Tag handling: allow overriding existing tags (default: preserve tags that existed before first run)
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_override_existing_tags',
            [
                'type' => 'boolean',
                'sanitize_callback' => function ($value) { return (bool) $value; },
                'default' => false,
            ]
        );
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_specs_tab_title',
            [
                'type' => 'string',
                'sanitize_callback' => function ($value) { return sanitize_text_field((string) $value); },
                'default' => __('Specifications', 'ai-product-attributes-for-woocommerce'),
            ]
        );
        // Attribute handling: allow overriding existing attributes (default: preserve attributes that existed before first run)
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_override_existing_attributes',
            [
                'type' => 'boolean',
                'sanitize_callback' => function ($value) { return (bool) $value; },
                'default' => false,
            ]
        );

        // Tabbed sections
        \add_settings_section(
            'dataalign_section_connection',
            \esc_html__('Connection', 'ai-product-attributes-for-woocommerce'),
            function () {
                echo '<p>' . \esc_html__('Configure your DataAlign API connection.', 'ai-product-attributes-for-woocommerce') . '</p>';
            },
            'dataalign-woo'
        );
        \add_settings_section(
            'dataalign_section_seo',
            \esc_html__('SEO', 'ai-product-attributes-for-woocommerce'),
            function () {
                echo '<p>' . \esc_html__('Control how enriched SEO maps into SEO plugins.', 'ai-product-attributes-for-woocommerce') . '</p>';
                echo '<p class="description">' . \esc_html__('Baseline & Restore: On first enrichment, we preserve your existing product SEO title and meta description from the active SEO plugin (Yoast, Rank Math, SEOPress, AIOSEO, or TSF). You can restore these preserved values anytime from the product page or via the bulk restore on this settings screen.', 'ai-product-attributes-for-woocommerce') . '</p>';
            },
            'dataalign-woo'
        );
        \add_settings_section(
            'dataalign_section_attributes',
            \esc_html__('Attributes & Specs', 'ai-product-attributes-for-woocommerce'),
            function () {
                echo '<p>' . \esc_html__('Sync attributes to WooCommerce and manage the Specs tab.', 'ai-product-attributes-for-woocommerce') . '</p>';
                echo '<p class="description">' . \esc_html__('Specs Tab shows a clean, structured table of your product’s attributes on the product page, built from DataAlign attributes. It helps shoppers compare products quickly and improves readability. You can enable/disable the tab and customize its title below. You can also embed the same table anywhere using the shortcode [dataalign_specs id="123"].', 'ai-product-attributes-for-woocommerce') . '</p>';
            },
            'dataalign-woo'
        );
        \add_settings_section(
            'dataalign_section_advanced',
            \esc_html__('Advanced', 'ai-product-attributes-for-woocommerce'),
            function () {
                echo '<p>' . \esc_html__('Advanced options for media and local development.', 'ai-product-attributes-for-woocommerce') . '</p>';
            },
            'dataalign-woo'
        );

        // API Base URL is fixed; no field rendered.

        \add_settings_field(
            'dataalign_api_key_field',
            \esc_html__('API Key', 'ai-product-attributes-for-woocommerce'),
            function () {
                $value = (string) \get_option('dataalign_api_key', '');
                // Use a neutral, random-looking placeholder so it doesn't imply other providers' key formats
                try {
                    $rand = bin2hex(random_bytes(12));
                } catch (\Throwable $e) {
                    $rand = bin2hex(uniqid('', true));
                }
                $placeholder = 'da_' . substr($rand, 0, 20);
                echo '<div class="dataalign-field">';
                echo '<input type="password" class="regular-text ltr" id="dataalign_api_key" name="dataalign_api_key" value="' . \esc_attr($value) . '" placeholder="' . \esc_attr($placeholder) . '" /> ';
                echo '<button type="button" class="button" id="dataalign-toggle-key" aria-controls="dataalign_api_key">' . \esc_html__('Show', 'ai-product-attributes-for-woocommerce') . '</button>';
                echo '</div>';
                echo '<p class="description">' . \esc_html__('Your secret API key. Keep it safe.', 'ai-product-attributes-for-woocommerce') . '</p>';
            },
            'dataalign-woo',
            'dataalign_section_connection'
        );

        \add_settings_field(
            'dataalign_public_base_url_field',
            \esc_html__('Public Site Base URL (optional)', 'ai-product-attributes-for-woocommerce'),
            function () {
                $value = (string) \get_option('dataalign_public_base_url', '');
                echo '<input type="url" class="regular-text ltr" id="dataalign_public_base_url" name="dataalign_public_base_url" value="' . \esc_attr($value) . '" placeholder="https://your-tunnel.example.com" />';
                echo '<p class="description">' . \esc_html__('If your site is local, set a public tunnel base (e.g., ngrok/Cloudflare) to rewrite media URLs sent to the API.', 'ai-product-attributes-for-woocommerce') . '</p>';
            },
            'dataalign-woo',
            'dataalign_section_advanced'
        );

        \add_settings_field(
            'dataalign_exclude_images_field',
            \esc_html__('Exclude Images from Enrichment', 'ai-product-attributes-for-woocommerce'),
            function () {
                $checked = (bool) \get_option('dataalign_exclude_images', false);
                echo '<label><input type="checkbox" id="dataalign_exclude_images" name="dataalign_exclude_images" value="1" ' . checked(true, $checked, false) . ' /> ' . \esc_html__('Do not send product images to the API (useful on local dev).', 'ai-product-attributes-for-woocommerce') . '</label>';
            },
            'dataalign-woo',
            'dataalign_section_advanced'
        );

        // Subheader: Attributes
        \add_settings_field(
            'dataalign_attributes_subheader',
            '',
            function () {
                echo '<h3 class="title" style="margin-top:0">' . \esc_html__('Attributes', 'ai-product-attributes-for-woocommerce') . '</h3>';
            },
            'dataalign-woo',
            'dataalign_section_attributes'
        );

        \add_settings_field(
            'dataalign_sync_attributes_to_woo_field',
            \esc_html__('Auto-sync Attributes to Woo', 'ai-product-attributes-for-woocommerce'),
            function () {
                $checked = (bool) \get_option('dataalign_sync_attributes_to_woo', true);
                echo '<label><input type="checkbox" id="dataalign_sync_attributes_to_woo" name="dataalign_sync_attributes_to_woo" value="1" ' . checked(true, $checked, false) . ' /> ' . \esc_html__('On enrichment/poll, copy DataAlign attributes into WooCommerce “Custom product attributes” (visible, non-variation). For variable products, attributes used for variations are never modified.', 'ai-product-attributes-for-woocommerce') . '</label>';
            },
            'dataalign-woo',
            'dataalign_section_attributes'
        );
        \add_settings_field(
            'dataalign_enable_specs_tab_field',
            \esc_html__('Enable Specs Tab', 'ai-product-attributes-for-woocommerce'),
            function () {
                $checked = (bool) \get_option('dataalign_enable_specs_tab', false);
                echo '<label><input type="checkbox" id="dataalign_enable_specs_tab" name="dataalign_enable_specs_tab" value="1" ' . checked(true, $checked, false) . ' /> ' . \esc_html__('Show a “Specifications” tab on product pages using DataAlign attributes.', 'ai-product-attributes-for-woocommerce') . '</label>';
            },
            'dataalign-woo',
            'dataalign_section_attributes'
        );
        \add_settings_field(
            'dataalign_specs_tab_title_field',
            \esc_html__('Specs Tab Title', 'ai-product-attributes-for-woocommerce'),
            function () {
                $value = (string) \get_option('dataalign_specs_tab_title', __('Specifications', 'ai-product-attributes-for-woocommerce'));
                echo '<input type="text" class="regular-text" id="dataalign_specs_tab_title" name="dataalign_specs_tab_title" value="' . \esc_attr($value) . '" />';
            },
            'dataalign-woo',
            'dataalign_section_attributes'
        );
        \add_settings_field(
            'dataalign_hide_additional_info_tab_field',
            \esc_html__('Hide “Additional information” Tab', 'ai-product-attributes-for-woocommerce'),
            function () {
                $checked = (bool) \get_option('dataalign_hide_additional_info_tab', false);
                echo '<label><input type="checkbox" id="dataalign_hide_additional_info_tab" name="dataalign_hide_additional_info_tab" value="1" ' . checked(true, $checked, false) . ' /> ' . \esc_html__('When the Specs tab is shown, remove WooCommerce’s “Additional information” tab to avoid duplicate content.', 'ai-product-attributes-for-woocommerce') . '</label>';
            },
            'dataalign-woo',
            'dataalign_section_attributes'
        );

        // Attributes override control
        \add_settings_field(
            'dataalign_override_existing_attributes_field',
            \esc_html__('Override Existing Attributes', 'ai-product-attributes-for-woocommerce'),
            function () {
                $checked = (bool) \get_option('dataalign_override_existing_attributes', false);
                echo '<label><input type="checkbox" id="dataalign_override_existing_attributes" name="dataalign_override_existing_attributes" value="1" ' . checked(true, $checked, false) . ' /> ' . \esc_html__('Replace previously existing attributes with AI-generated values. When disabled (default), attributes that existed before the first enrichment are preserved.', 'ai-product-attributes-for-woocommerce') . '</label>';
                echo '<p class="description">' . \esc_html__('Note: We snapshot your pre‑AI attributes on the first enrichment. That baseline is always preserved and can be restored from the product edit screen (DataAlign box).', 'ai-product-attributes-for-woocommerce') . '</p>';
            },
            'dataalign-woo',
            'dataalign_section_attributes'
        );

        // Subheader: Tags
        \add_settings_field(
            'dataalign_tags_subheader',
            '',
            function () {
                echo '<h3 class="title" style="margin-top:16px">' . \esc_html__('Tags', 'ai-product-attributes-for-woocommerce') . '</h3>';
            },
            'dataalign-woo',
            'dataalign_section_attributes'
        );
        // Tags setting appears after attributes-related settings
        \add_settings_field(
            'dataalign_override_existing_tags_field',
            \esc_html__('Override Existing Product Tags', 'ai-product-attributes-for-woocommerce'),
            function () {
                $checked = (bool) \get_option('dataalign_override_existing_tags', false);
                echo '<label><input type="checkbox" id="dataalign_override_existing_tags" name="dataalign_override_existing_tags" value="1" ' . checked(true, $checked, false) . ' /> ' . \esc_html__('Replace all product tags with AI-generated tags. When disabled (default), tags that existed before the first enrichment are preserved.', 'ai-product-attributes-for-woocommerce') . '</label>';
                echo '<p class="description">' . \esc_html__('Note: We snapshot your pre‑AI tags on the first enrichment. That baseline is always preserved and can be restored from the product edit screen (DataAlign box).', 'ai-product-attributes-for-woocommerce') . '</p>';
                echo '<p class="description">' . \esc_html__('Best practice: The plugin applies at most 5 product tags per item.', 'ai-product-attributes-for-woocommerce') . '</p>';
            },
            'dataalign-woo',
            'dataalign_section_attributes'
        );

        // SEO mapping settings
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_map_seo_to_plugins',
            [
                'type' => 'boolean',
                'sanitize_callback' => function ($value) { return (bool) $value; },
                'default' => false,
            ]
        );
        \register_setting(
            'dataalign_woo_settings',
            'dataalign_seo_fill_only_empty',
            [
                'type' => 'boolean',
                'sanitize_callback' => function ($value) { return (bool) $value; },
                'default' => true,
            ]
        );

        \add_settings_field(
            'dataalign_map_seo_to_plugins_field',
            \esc_html__('Map SEO to SEO Plugins', 'ai-product-attributes-for-woocommerce'),
            function () {
                $checked = (bool) \get_option('dataalign_map_seo_to_plugins', false);
                echo '<label><input type="checkbox" id="dataalign_map_seo_to_plugins" name="dataalign_map_seo_to_plugins" value="1" ' . checked(true, $checked, false) . ' /> ' . \esc_html__('Copy enriched SEO title/description into popular SEO plugins (Yoast, Rank Math, SEOPress, AIOSEO, TSF).', 'ai-product-attributes-for-woocommerce') . '</label>';
            },
            'dataalign-woo',
            'dataalign_section_seo'
        );

        \add_settings_field(
            'dataalign_seo_fill_only_empty_field',
            \esc_html__('Only Fill Empty SEO Fields', 'ai-product-attributes-for-woocommerce'),
            function () {
                $checked = (bool) \get_option('dataalign_seo_fill_only_empty', true);
                echo '<label><input type="checkbox" id="dataalign_seo_fill_only_empty" name="dataalign_seo_fill_only_empty" value="1" ' . checked(true, $checked, false) . ' /> ' . \esc_html__('When mapping to SEO plugins, do not overwrite existing per-post SEO values.', 'ai-product-attributes-for-woocommerce') . '</label>';
            },
            'dataalign-woo',
            'dataalign_settings_section'
        );
    }

    /**
     * Render the settings page contents.
     */
    public static function render_settings_page(): void
    {
        if (!\current_user_can('manage_woocommerce')) {
            \wp_die(\esc_html__('You do not have permission to access this page.', 'ai-product-attributes-for-woocommerce'));
        }
        echo '<div class="wrap dataalign-settings">';
        echo '<h1 class="wp-heading-inline">' . \esc_html__('DataAlign AI Settings', 'ai-product-attributes-for-woocommerce') . '</h1>';
        // Hero / Overview
        echo '<div class="dataalign-hero dataalign-reveal">';
        echo '  <div class="dataalign-hero-content">';
        echo '    <div class="dataalign-badges">';
        echo '      <span class="dataalign-badge badge-green">' . \esc_html__('WooCommerce Ready', 'ai-product-attributes-for-woocommerce') . '</span>';
        echo '      <span class="dataalign-badge badge-purple">' . \esc_html__('AI-Powered', 'ai-product-attributes-for-woocommerce') . '</span>';
        echo '    </div>';
        echo '    <h2>' . \esc_html__('Enrich and Align Your Product Data with AI', 'ai-product-attributes-for-woocommerce') . '</h2>';
        echo '    <svg class="dataalign-ai-graphic" width="72" height="32" viewBox="0 0 72 32" fill="none" xmlns="http://www.w3.org/2000/svg" aria-hidden="true">'
            . '<rect x="1" y="6" width="46" height="20" rx="4" stroke="white" stroke-opacity="0.8"/>'
            . '<circle cx="54" cy="8" r="3" stroke="white" stroke-opacity="0.8"/>'
            . '<circle cx="62" cy="16" r="3" stroke="white" stroke-opacity="0.8"/>'
            . '<circle cx="54" cy="24" r="3" stroke="white" stroke-opacity="0.8"/>'
            . '<path d="M47 8 H51" stroke="white" stroke-opacity="0.8"/>'
            . '<path d="M47 24 H51" stroke="white" stroke-opacity="0.8"/>'
            . '<path d="M47 16 H59" stroke="white" stroke-opacity="0.8"/>'
            . '</svg>';
        echo '    <p class="dataalign-hero-sub">' . \esc_html__('One-click enrichment for tags, specs, and SEO. Queue bulk jobs and let DataAlign handle the rest — automatically.', 'ai-product-attributes-for-woocommerce') . '</p>';
        echo '    <p><a class="dataalign-hero-link" href="' . \esc_url('https://dataalign.ai/') . '" target="_blank" rel="noopener noreferrer">' . \esc_html__('Learn more at DataAlign.ai →', 'ai-product-attributes-for-woocommerce') . '</a></p>';
        echo '  </div>';
        echo '</div>';

        // Minimum requirement + accuracy tip
        echo '<div class="notice notice-warning"><p>'
            . \esc_html__('Minimum requirement: each product must have a Title. DataAlign uses the product title as the baseline input for enrichment.', 'ai-product-attributes-for-woocommerce')
            . ' '
            . \esc_html__('Tip: For best accuracy, provide a clear, detailed product description.', 'ai-product-attributes-for-woocommerce')
            . ' '
            . \esc_html__('Images can further improve accuracy but are optional.', 'ai-product-attributes-for-woocommerce')
            . '</p></div>';

        echo '<div class="metabox-holder">';

        // API Configuration Box
        echo '<div class="postbox">';
        echo '<h2 class="hndle"><span class="dashicons dashicons-admin-generic" style="margin-right:6px"></span><span>' . \esc_html__('API Configuration', 'ai-product-attributes-for-woocommerce') . '</span></h2>';
        echo '<div class="inside">';
        echo '<form method="post" action="options.php" class="dataalign-tabs-form">';
        \settings_fields('dataalign_woo_settings');
        echo '<h2 class="nav-tab-wrapper dataalign-tabs-nav">';
        echo '  <a href="#connection" class="nav-tab nav-tab-active" data-tab="connection">' . \esc_html__('Connection', 'ai-product-attributes-for-woocommerce') . '</a>';
        echo '  <a href="#attributes" class="nav-tab" data-tab="attributes">' . \esc_html__('Attributes & Specs', 'ai-product-attributes-for-woocommerce') . '</a>';
        echo '  <a href="#seo" class="nav-tab" data-tab="seo">' . \esc_html__('SEO', 'ai-product-attributes-for-woocommerce') . '</a>';
        echo '  <a href="#optimize" class="nav-tab" data-tab="optimize">' . \esc_html__('Optimize', 'ai-product-attributes-for-woocommerce') . '</a>';
        echo '  <a href="#restore" class="nav-tab" data-tab="restore">' . \esc_html__('Restore', 'ai-product-attributes-for-woocommerce') . '</a>';
        echo '  <a href="#advanced" class="nav-tab" data-tab="advanced">' . \esc_html__('Advanced', 'ai-product-attributes-for-woocommerce') . '</a>';
        echo '</h2>';
        // Connection tab
        echo '<div class="dataalign-tab-panel" id="tab-connection">';
        echo '  <table class="form-table" role="presentation">';
        \do_settings_fields('dataalign-woo', 'dataalign_section_connection');
        echo '  </table>';
        echo '  <p class="submit">';
        \submit_button(\esc_html__('Save Settings', 'ai-product-attributes-for-woocommerce'), 'primary', 'submit', false);
        $has_key = (string) \get_option('dataalign_api_key', '') !== '';
        // Render Test button but hide it until a key exists, so it can appear immediately after registration
        $test_style_attr = $has_key ? '' : 'display:none';
        echo ' <button type="button" id="dataalign-test-connection" class="button"' . ($test_style_attr !== '' ? ' style="' . \esc_attr($test_style_attr) . '"' : '') . '>' . \esc_html__('Test Connection', 'ai-product-attributes-for-woocommerce') . '</button>';
        if (!$has_key) {
            echo ' <button type="button" id="dataalign-register-site" class="button button-secondary">' . \esc_html__('Register Site (Get Free API Key)', 'ai-product-attributes-for-woocommerce') . '</button>';
        }
        echo ' <span class="spinner" id="dataalign-settings-spinner" style="float:none"></span>';
        echo ' <span id="dataalign-test-result" class="dataalign-inline-status"></span>';
        echo '  </p>';
        echo '</div>';
        // Optimize tab
        echo '<div class="dataalign-tab-panel hidden" id="tab-optimize">';
        echo '  <div class="postbox" style="margin-top:12px">';
        echo '    <h2 class="hndle"><span class="dashicons dashicons-performance" style="margin-right:6px"></span><span>' . \esc_html__('Bulk Optimization (All Products)', 'ai-product-attributes-for-woocommerce') . '</span></h2>';
        echo '    <div class="inside">';
        echo '      <p>' . \esc_html__('Queue enrichment jobs for all products. Keep this Optimize tab open to queue in real time (50 per tick). Updating products runs in the background once queuing completes.', 'ai-product-attributes-for-woocommerce') . '</p>';
        echo '      <p>';
        echo '        <button type="button" class="button button-primary" id="dataalign-start-bulk-enrich">' . \esc_html__('Start Bulk Optimization', 'ai-product-attributes-for-woocommerce') . '</button> ';
        echo '        <button type="button" class="button" id="dataalign-cancel-bulk-enrich" disabled>' . \esc_html__('Cancel', 'ai-product-attributes-for-woocommerce') . '</button> ';
        echo '        <button type="button" class="button" id="dataalign-poll-now">' . \esc_html__('Poll Now', 'ai-product-attributes-for-woocommerce') . '</button> ';
        echo '        <span class="spinner" id="dataalign-bulk-enrich-spinner" style="float:none"></span>';
        echo '        <span id="dataalign-bulk-enrich-status" class="dataalign-inline-status"></span>';
        echo '      </p>';
        echo '      <p id="dataalign-bulk-background-note" class="description" style="margin-top:6px; display:none">' . \esc_html__('Bulk optimization runs in the background even if you leave this page or close your browser. You can return here later to check progress.', 'ai-product-attributes-for-woocommerce') . '</p>';
        echo '      <div style="margin-top:10px; max-width:520px">';
        echo '        <p style="margin:6px 0 4px"><strong>' . \esc_html__('Queuing products', 'ai-product-attributes-for-woocommerce') . '</strong> <small id="dataalign-bulk-enrich-queue-text"></small></p>';
        echo '        <div style="width:100%; background:#f0f0f1; border-radius:4px; height:10px; overflow:hidden" aria-hidden="false" aria-label="' . \esc_attr__('Queuing progress', 'ai-product-attributes-for-woocommerce') . '">';
        echo '          <div id="dataalign-bulk-enrich-queue-bar" style="width:0%; height:10px; background:#2271b1" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0"></div>';
        echo '        </div>';
        echo '        <p style="margin:10px 0 4px"><strong>' . \esc_html__('Updating products', 'ai-product-attributes-for-woocommerce') . '</strong> <small id="dataalign-bulk-enrich-apply-text"></small></p>';
        echo '        <div style="width:100%; background:#f0f0f1; border-radius:4px; height:10px; overflow:hidden" aria-hidden="false" aria-label="' . \esc_attr__('Update progress', 'ai-product-attributes-for-woocommerce') . '">';
        echo '          <div id="dataalign-bulk-enrich-apply-bar" style="width:0%; height:10px; background:#4ab866" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0"></div>';
        echo '        </div>';
        echo '      </div>';
        // Live logs box
        echo '      <div style="margin-top:12px; max-width:620px">';
        echo '        <h3 class="title" style="margin:4px 0">' . \esc_html__('Live Logs', 'ai-product-attributes-for-woocommerce') . '</h3>';
        echo '        <div id="dataalign-live-logs" style="max-height:180px;overflow:auto;background:#0b1a2b;border:1px solid #1f2937;padding:8px;border-radius:6px;font-family:monospace;font-size:12px;color:#cbd5e1"></div>';
        echo '      </div>';
        echo '    </div>';
        echo '  </div>';
        echo '</div>';
        // Attributes & Specs tab (second)
        echo '<div class="dataalign-tab-panel hidden" id="tab-attributes">';
        echo '  <table class="form-table" role="presentation">';
        \do_settings_fields('dataalign-woo', 'dataalign_section_attributes');
        echo '  </table>';
        echo '  <p class="submit">';
        \submit_button(\esc_html__('Save Settings', 'ai-product-attributes-for-woocommerce'), 'primary', 'submit', false);
        echo '  </p>';
        echo '</div>';
        // SEO tab (third)
        echo '<div class="dataalign-tab-panel hidden" id="tab-seo">';
        echo '  <table class="form-table" role="presentation">';
        \do_settings_fields('dataalign-woo', 'dataalign_section_seo');
        echo '  </table>';
        echo '  <p class="submit">';
        \submit_button(\esc_html__('Save Settings', 'ai-product-attributes-for-woocommerce'), 'primary', 'submit', false);
        echo '  </p>';
        echo '</div>';
        // Restore tab
        echo '<div class="dataalign-tab-panel hidden" id="tab-restore">';
        echo '  <div class="postbox" style="margin-top:12px">';
        echo '    <h2 class="hndle"><span class="dashicons dashicons-backup" style="margin-right:6px"></span><span>' . \esc_html__('Restore Baseline Tags, Attributes & SEO (All Products)', 'ai-product-attributes-for-woocommerce') . '</span></h2>';
        echo '    <div class="inside">';
        echo '      <p>' . \esc_html__('Reapply the pre‑AI tags, attributes, and SEO captured on the first enrichment for all products that have a recorded baseline.', 'ai-product-attributes-for-woocommerce') . '</p>';
        echo '      <p>';
        echo '        <button type="button" class="button" id="dataalign-restore-all-attributes">' . \esc_html__('Restore pre‑AI Tags, Attributes & SEO (All)', 'ai-product-attributes-for-woocommerce') . '</button> ';
        echo '        <span class="spinner" id="dataalign-restore-all-spinner" style="float:none"></span>';
        echo '        <span id="dataalign-restore-all-status" class="dataalign-inline-status"></span>';
        echo '      </p>';
        echo '      <div style="margin-top:6px; max-width:480px; background:#f0f0f1; border-radius:4px; height:10px; overflow:hidden" aria-hidden="false" aria-label="' . \esc_attr__('Restore progress', 'ai-product-attributes-for-woocommerce') . '">';
        echo '        <div id="dataalign-restore-all-bar" style="width:0%; height:10px; background:#1e8cbe" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0"></div>';
        echo '      </div>';
        echo '    </div>';
        echo '  </div>';
        echo '</div>';

        // Advanced tab
        echo '<div class="dataalign-tab-panel hidden" id="tab-advanced">';
        echo '  <table class="form-table" role="presentation">';
        \do_settings_fields('dataalign-woo', 'dataalign_section_advanced');
        echo '  </table>';
        echo '  <p class="submit">';
        \submit_button(\esc_html__('Save Settings', 'ai-product-attributes-for-woocommerce'), 'primary', 'submit', false);
        echo '  </p>';
        echo '</div>';

        echo '</form>';
        echo '</div>';
        echo '</div>';

        // Features Box
        echo '<div class="postbox dataalign-reveal">';
        echo '<h2 class="hndle"><span class="dashicons dashicons-lightbulb" style="margin-right:6px"></span><span>' . \esc_html__('Highlights', 'ai-product-attributes-for-woocommerce') . '</span></h2>';
        echo '<div class="inside">';
        echo '  <div class="dataalign-card-grid">';
        echo '    <div class="dataalign-card">';
        echo '      <span class="dashicons dashicons-yes-alt"></span>';
        echo '      <h3>' . \esc_html__('Single-Product Enrichment', 'ai-product-attributes-for-woocommerce') . '</h3>';
        echo '      <p>' . \esc_html__('Click the Enrich button in any product to generate tags, specs, and SEO.', 'ai-product-attributes-for-woocommerce') . '</p>';
        echo '    </div>';
        echo '    <div class="dataalign-card">';
        echo '      <span class="dashicons dashicons-update"></span>';
        echo '      <h3>' . \esc_html__('Bulk Queue', 'ai-product-attributes-for-woocommerce') . '</h3>';
        echo '      <p>' . \esc_html__('Run enrichment for many products at once. We store job IDs and poll automatically.', 'ai-product-attributes-for-woocommerce') . '</p>';
        echo '    </div>';
        echo '    <div class="dataalign-card">';
        echo '      <span class="dashicons dashicons-clock"></span>';
        echo '      <h3>' . \esc_html__('Auto Polling', 'ai-product-attributes-for-woocommerce') . '</h3>';
        echo '      <p>' . \esc_html__('WP-Cron checks for completed jobs and applies results with no extra clicks.', 'ai-product-attributes-for-woocommerce') . '</p>';
        echo '    </div>';
        echo '    <div class="dataalign-card">';
        echo '      <span class="dashicons dashicons-search"></span>';
        echo '      <h3>' . \esc_html__('SEO Meta', 'ai-product-attributes-for-woocommerce') . '</h3>';
        echo '      <p>' . \esc_html__('Stores enriched SEO title and description. Optionally maps into Yoast, Rank Math, SEOPress, AIOSEO, and TSF. Preserves your pre‑AI SEO and supports one‑click restore.', 'ai-product-attributes-for-woocommerce') . '</p>';
        echo '    </div>';
        echo '  </div>';
        echo '</div>';
        echo '</div>';

        // Usage display Box
        $api_key  = (string) \get_option('dataalign_api_key', '');
        $configured = ($api_key !== '');
        echo '<div class="postbox dataalign-reveal">';
        echo '<h2 class="hndle"><span class="dashicons dashicons-chart-area" style="margin-right:6px"></span><span>' . \esc_html__('Usage', 'ai-product-attributes-for-woocommerce') . '</span></h2>';
        echo '<div class="inside">';
        if ($configured) {
            echo '<div id="dataalign-usage-container"><p>' . \esc_html__('Loading usage…', 'ai-product-attributes-for-woocommerce') . '</p></div>';
        } else {
            echo '<p>' . \esc_html__('Configure your API Key to view usage.', 'ai-product-attributes-for-woocommerce') . '</p>';
        }
        echo '</div>';
        echo '</div>';

        // Diagnostics Box
        echo '<div class="postbox dataalign-reveal">';
        echo '<h2 class="hndle"><span class="dashicons dashicons-admin-tools" style="margin-right:6px"></span><span>' . \esc_html__('Diagnostics', 'ai-product-attributes-for-woocommerce') . '</span></h2>';
        echo '<div class="inside">';
        $last_global_error = (string) \get_option('_dataalign_last_global_error', '');
        if ($last_global_error) {
            echo '<div class="notice notice-error"><p>' . \esc_html($last_global_error) . '</p></div>';
        } else {
            echo '<p>' . \esc_html__('No recent errors recorded.', 'ai-product-attributes-for-woocommerce') . '</p>';
        }
        echo '</div>';
        echo '</div>';

        // How it works Box
        echo '<div class="postbox dataalign-reveal">';
        echo '<h2 class="hndle"><span class="dashicons dashicons-admin-site" style="margin-right:6px"></span><span>' . \esc_html__('How It Works', 'ai-product-attributes-for-woocommerce') . '</span></h2>';
        echo '<div class="inside">';
        echo '  <ol class="dataalign-steps">';
        echo '    <li><span class="step">1</span> ' . \esc_html__('Set your API Key below, then Test Connection.', 'ai-product-attributes-for-woocommerce') . '</li>';
        echo '    <li><span class="step">2</span> ' . \esc_html__('Open a product and click Enrich to generate tags, specs, and SEO.', 'ai-product-attributes-for-woocommerce') . '</li>';
        echo '    <li><span class="step">3</span> ' . \esc_html__('Use Bulk Enrich from Products list for many items.', 'ai-product-attributes-for-woocommerce') . '</li>';
        echo '    <li><span class="step">4</span> ' . \esc_html__('WP-Cron applies results as jobs complete — check Diagnostics if something fails.', 'ai-product-attributes-for-woocommerce') . '</li>';
        echo '  </ol>';
        echo '</div>';
        echo '</div>';

        echo '</div>';
        echo '</div>';
    }

    /**
     * Display admin notices for configuration and global errors.
     */
    public static function admin_notices(): void
    {
        if (!\current_user_can('manage_woocommerce')) {
            return;
        }

        $screen = function_exists('get_current_screen') ? \get_current_screen() : null;
        // Always show on product and WooCommerce pages; otherwise show globally to admins

        $api_key  = (string) \get_option('dataalign_api_key', '');
        if ($api_key === '') {
            $settings_url = \admin_url('admin.php?page=dataalign-woo');
            echo '<div class="notice notice-warning"><p>'
                . \esc_html__('DataAlign API is not configured. Please set your API key in WooCommerce → DataAlign AI.', 'ai-product-attributes-for-woocommerce')
                . ' <a href="' . \esc_url($settings_url) . '">' . \esc_html__('Open settings', 'ai-product-attributes-for-woocommerce') . '</a>'
                . '</p></div>';
        }

        // Optional: show last global error if present, with secure dismiss link
        $last_error = (string) \get_option('_dataalign_last_global_error', '');
        if ($last_error !== '') {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $dismiss = isset($_GET['dataalign_dismiss_global_error']) ? sanitize_text_field( wp_unslash((string) $_GET['dataalign_dismiss_global_error']) ) : '';
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $nonce = isset($_GET['_dataalign_nonce']) ? sanitize_text_field( wp_unslash((string) $_GET['_dataalign_nonce']) ) : '';
            if ($dismiss === '1' && \wp_verify_nonce($nonce, 'dataalign_dismiss_global_error')) {
                \delete_option('_dataalign_last_global_error');
            } else {
                $dismiss_url = \wp_nonce_url(\add_query_arg('dataalign_dismiss_global_error', '1'), 'dataalign_dismiss_global_error', '_dataalign_nonce');
                $msg = self::shorten_error_message($last_error);
                echo '<div class="notice notice-error is-dismissible"><p>' . \esc_html($msg) . ' '
                    . '<a href="' . \esc_url($dismiss_url) . '">' . \esc_html__('Dismiss', 'ai-product-attributes-for-woocommerce') . '</a>'
                    . '</p></div>';
            }
        }
    }

    /**
     * Reduce noisy stack traces from remote responses for admin notices.
     */
    protected static function shorten_error_message(string $msg): string
    {
        $clean = wp_strip_all_tags($msg);
        // If the message looks like a long stack trace, keep the first 200 chars.
        if (strlen($clean) > 220) {
            $clean = substr($clean, 0, 200) . '…';
        }
        // Prefer the part before 'Stack trace' if present.
        $parts = preg_split('/Stack trace:/i', $clean);
        if (is_array($parts) && !empty($parts[0])) {
            $first = trim($parts[0]);
            if ($first !== '') { return $first; }
        }
        return trim($clean);
    }

    /**
     * Make minimum requirements obvious on product edit screens.
     */
    public static function product_requirements_notices(): void
    {
        if (!\current_user_can('edit_products')) {
            return;
        }
        $screen = function_exists('get_current_screen') ? \get_current_screen() : null;
        if (!$screen || $screen->id !== 'product') {
            return;
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $post_id = isset($_GET['post']) ? (int) $_GET['post'] : 0;
        if ($post_id <= 0) {
            // New product screen: still show requirement
            echo '<div class="notice notice-warning"><p><strong>'
                . \esc_html__('Required:', 'ai-product-attributes-for-woocommerce') . '</strong> '
                . \esc_html__('Each product must have a Title for DataAlign enrichment to work.', 'ai-product-attributes-for-woocommerce')
                . ' '
                . \esc_html__('Tip: For best accuracy, provide a clear, detailed product description.', 'ai-product-attributes-for-woocommerce')
                . ' '
                . \esc_html__('Images can further improve accuracy but are optional.', 'ai-product-attributes-for-woocommerce')
                . '</p></div>';
            return;
        }
        $post = \get_post($post_id);
        if (!$post || $post->post_type !== 'product') {
            return;
        }
        $title_empty = trim((string) $post->post_title) === '';
        if ($title_empty) {
            echo '<div class="notice notice-warning"><p><strong>'
                . \esc_html__('Required:', 'ai-product-attributes-for-woocommerce') . '</strong> '
                . \esc_html__('Each product must have a Title for DataAlign enrichment to work.', 'ai-product-attributes-for-woocommerce')
                . ' '
                . \esc_html__('Tip: For best accuracy, provide a clear, detailed product description.', 'ai-product-attributes-for-woocommerce')
                . '</p></div>';
        } else {
            echo '<div class="notice notice-info"><p><strong>'
                . \esc_html__('Tip:', 'ai-product-attributes-for-woocommerce') . '</strong> '
                . \esc_html__('For best accuracy, provide a clear, detailed product description.', 'ai-product-attributes-for-woocommerce')
                . ' '
                . \esc_html__('Images can further improve accuracy but are optional.', 'ai-product-attributes-for-woocommerce')
                . '</p></div>';
        }
    }

    /**
     * AJAX: Test connection using get_usage() to validate credentials quickly.
     */
    public static function ajax_test_connection(): void
    {
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }

        $client = new DataAlign_API_Client();
        $usage  = $client->get_usage();
        if (\is_wp_error($usage)) {
            \wp_send_json_error(['message' => \esc_html($usage->get_error_message())]);
        }
        $data = isset($usage['data']) && is_array($usage['data']) ? $usage['data'] : [];
        \wp_send_json_success([
            'message' => \esc_html__('Connection successful.', 'ai-product-attributes-for-woocommerce'),
            'data'    => $data,
        ]);
    }

    /**
     * AJAX: Fetch usage info (lazy-loaded on settings page).
     */
    public static function ajax_get_usage(): void
    {
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }

        $api_key  = (string) \get_option('dataalign_api_key', '');
        if ($api_key === '') {
            \wp_send_json_error(['message' => \esc_html__('API not configured.', 'ai-product-attributes-for-woocommerce')]);
        }

        $client = new DataAlign_API_Client();
        $usage  = $client->get_usage();
        if (\is_wp_error($usage)) {
            \wp_send_json_error(['message' => \esc_html($usage->get_error_message())]);
        }
        $data = isset($usage['data']) && is_array($usage['data']) ? $usage['data'] : [];
        \wp_send_json_success([
            'yearMonth'         => isset($data['yearMonth']) ? (string) $data['yearMonth'] : '',
            'productsProcessed' => isset($data['productsProcessed']) ? (int) $data['productsProcessed'] : 0,
            'limit'             => isset($data['limit']) ? (int) $data['limit'] : 0,
            'planName'          => isset($data['planName']) ? (string) $data['planName'] : '',
        ]);
    }

    /**
     * AJAX: Register site and save API key.
     */
    public static function ajax_register_site(): void
    {
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }

        // Base URL is fixed; no preliminary check needed.

        // Match OpenAPI: /api/v1/register-site expects store_name, store_url, platform
        $payload = [
            'store_name' => (string) get_bloginfo('name'),
            'store_url'  => (string) home_url('/'),
            'platform'   => 'wordpress',
        ];

        $client = new DataAlign_API_Client();
        $resp = $client->register_site($payload);
        if (\is_wp_error($resp)) {
            \wp_send_json_error(['message' => \esc_html($resp->get_error_message())]);
        }
        $success = isset($resp['success']) ? (bool) $resp['success'] : false;
        $data    = isset($resp['data']) && is_array($resp['data']) ? $resp['data'] : [];
        if (!$success || empty($data)) {
            $msg = isset($resp['message']) ? (string) $resp['message'] : __('Registration failed.', 'ai-product-attributes-for-woocommerce');
            \wp_send_json_error(['message' => \esc_html($msg)]);
        }

        $api_key = isset($data['api_key']) ? (string) $data['api_key'] : '';
        if ($api_key !== '') {
            \update_option('dataalign_api_key', $api_key);
        }
        if (isset($data['site_id'])) { \update_option('_dataalign_site_id', (int) $data['site_id']); }
        if (isset($data['plan']) && \is_array($data['plan'])) {
            $plan = $data['plan'];
            if (isset($plan['name'])) { \update_option('_dataalign_plan_name', (string) $plan['name']); }
            if (isset($plan['limit'])) { \update_option('_dataalign_plan_limit', (int) $plan['limit']); }
        }

        \wp_send_json_success([
            'message' => \esc_html__('Site registered and API key saved.', 'ai-product-attributes-for-woocommerce'),
            'data'    => [
                'api_key' => $api_key,
                'site_id' => isset($data['site_id']) ? (int) $data['site_id'] : 0,
                'plan'    => isset($data['plan']) && is_array($data['plan']) ? $data['plan'] : new \stdClass(),
            ],
        ]);
    }

    /**
     * AJAX: Sync stored DataAlign attributes to WooCommerce custom product attributes.
     */
    public static function ajax_sync_attributes(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $product_id = isset($_POST['product_id']) ? (int) $_POST['product_id'] : 0;
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if ($product_id <= 0 || !\wp_verify_nonce($nonce, 'dataalign_sync_' . $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('edit_product', $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        $attrs = \get_post_meta($product_id, '_dataalign_attributes', true);
        if (!is_array($attrs) || empty($attrs)) {
            \wp_send_json_error(['message' => \esc_html__('No DataAlign attributes to sync.', 'ai-product-attributes-for-woocommerce')]);
        }
        // Use Override Existing Attributes to determine fill behavior for Woo sync
        $fill_only = ! (bool) \get_option('dataalign_override_existing_attributes', false);
        $applied = self::sync_attributes_to_woo($product_id, $attrs, $fill_only);
        if (\is_wp_error($applied)) {
            \wp_send_json_error(['message' => \esc_html($applied->get_error_message())]);
        }
        \wp_send_json_success(['message' => \esc_html__('Attributes synced.', 'ai-product-attributes-for-woocommerce')]);
    }

    /**
     * Copy key/value attributes into WooCommerce product custom attributes (visible, non-variation).
     *
     * @param int   $product_id
     * @param array $attrs      key => value pairs
     * @param bool  $only_fill  if true, do not overwrite existing attribute values
     * @return bool|\WP_Error
     */
    protected static function sync_attributes_to_woo(int $product_id, array $attrs, bool $only_fill = true)
    {
        if (!function_exists('wc_get_product')) {
            return new \WP_Error('dataalign_wc_missing', __('WooCommerce is required.', 'ai-product-attributes-for-woocommerce'));
        }
        $product = \wc_get_product($product_id);
        if (!$product) {
            return new \WP_Error('dataalign_product_missing', __('Product not found.', 'ai-product-attributes-for-woocommerce'));
        }

        $meta = \get_post_meta($product_id, '_product_attributes', true);
        if (!is_array($meta)) {
            $meta = [];
        }

        // Determine attribute keys and slugs used for variations; never modify or duplicate these.
        $variation_keys = [];
        $variation_slugs = [];
        foreach ($meta as $k => $entry) {
            $is_var = isset($entry['is_variation']) ? (int) $entry['is_variation'] : 0;
            if ($is_var === 1) {
                $variation_keys[(string) $k] = true;
                // Collect normalized slugs for safer comparisons (e.g., 'pa_color' -> 'color', 'Color' -> 'color')
                $name_field = isset($entry['name']) ? (string) $entry['name'] : '';
                if ($name_field !== '') {
                    $slug = strtolower(trim($name_field));
                    if (strpos($slug, 'pa_') === 0) { $slug = substr($slug, 3); }
                    $slug = sanitize_title($slug);
                    if ($slug !== '') { $variation_slugs[$slug] = true; }
                }
                // Also include key-derived slug
                $k_slug = strtolower((string) $k);
                if (strpos($k_slug, 'attribute_') === 0) { $k_slug = substr($k_slug, 10); }
                if (strpos($k_slug, 'pa_') === 0) { $k_slug = substr($k_slug, 3); }
                $k_slug = sanitize_title($k_slug);
                if ($k_slug !== '') { $variation_slugs[$k_slug] = true; }
            }
        }

        // Extra safety: also detect variation attributes from the product object
        if (method_exists($product, 'get_attributes')) {
            try {
                $atts = (array) $product->get_attributes();
                foreach ($atts as $att) {
                    if (is_object($att) && method_exists($att, 'get_variation') && $att->get_variation()) {
                        $nm = '';
                        if (method_exists($att, 'get_name')) { $nm = (string) $att->get_name(); }
                        $slug = strtolower(trim($nm));
                        if ($slug !== '') {
                            if (strpos($slug, 'pa_') === 0) { $slug = substr($slug, 3); }
                            $slug = sanitize_title($slug);
                            if ($slug !== '') { $variation_slugs[$slug] = true; }
                        }
                    }
                }
            } catch (\Throwable $e) { /* ignore */ }
        }

        // Prepare incoming entries once
        $incoming = [];
        foreach ($attrs as $k => $v) {
            $name = trim((string) $k);
            $value = trim(is_array($v) ? implode(', ', array_filter(array_map('strval', $v))) : (string) $v);
            if ($name === '' || $value === '') { continue; }
            $base_slug = sanitize_title($name);
            $key = 'attribute_' . $base_slug;
            // Skip if this key corresponds to or collides with a variation attribute
            if (isset($variation_keys[$key])) { continue; }
            if (isset($variation_slugs[$base_slug])) { continue; }
            // Also skip if a corresponding taxonomy variation key exists (pa_{slug})
            if (isset($variation_keys['pa_' . $base_slug])) { continue; }
            $incoming[$key] = [
                'name'         => $name,
                'value'        => $value,
                'position'     => isset($meta[$key]['position']) ? (int) $meta[$key]['position'] : count($meta),
                'is_visible'   => 1,
                'is_variation' => 0,
                'is_taxonomy'  => 0,
            ];
        }

        if ($only_fill) {
            // Add only where empty or missing; preserve everything else
            foreach ($incoming as $key => $entry) {
                // Never touch variation attributes
                if (isset($variation_keys[$key])) { continue; }
                if (isset($meta[$key]) && !empty($meta[$key]['value'])) {
                    continue;
                }
                $meta[$key] = $entry;
            }
        } else {
            // Replace all non-taxonomy attributes with the incoming set. Preserve taxonomy-based attributes.
            $kept = [];
            foreach ($meta as $key => $entry) {
                $is_tax = isset($entry['is_taxonomy']) ? (int) $entry['is_taxonomy'] : 0;
                $is_var = isset($entry['is_variation']) ? (int) $entry['is_variation'] : 0;
                // Keep taxonomy attributes and any attributes used for variations
                if ($is_tax === 1 || $is_var === 1) {
                    $kept[$key] = $entry;
                }
            }
            // Also ensure we never add incoming entries that would target variation keys
            $incoming_filtered = [];
            foreach ($incoming as $k => $entry) {
                if (isset($variation_keys[$k])) { continue; }
                // Avoid adding an incoming attribute that collides by slug with a variation attribute
                $ik = strtolower((string) $k);
                if (strpos($ik, 'attribute_') === 0) { $ik = substr($ik, 10); }
                $ik = sanitize_title($ik);
                if ($ik !== '' && isset($variation_slugs[$ik])) { continue; }
                if (isset($variation_keys['pa_' . $ik])) { continue; }
                $incoming_filtered[$k] = $entry;
            }
            $meta = $kept + $incoming_filtered;
            // Recompute positions sequentially for consistency
            $pos = 0;
            foreach ($meta as $k => $entry) {
                $entry['position'] = $pos++;
                $meta[$k] = $entry;
            }
        }

        \update_post_meta($product_id, '_product_attributes', $meta);
        return true;
    }

    /**
     * AJAX: Restore tags and attributes to baseline captured on first enrichment.
     */
    public static function ajax_restore_baseline(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $product_id = isset($_POST['product_id']) ? (int) $_POST['product_id'] : 0;
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if ($product_id <= 0 || !\wp_verify_nonce($nonce, 'dataalign_restore_' . $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('edit_product', $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }

        $baseline_tags = \get_post_meta($product_id, '_dataalign_preserved_tag_ids', true);
        $baseline_attrs = \get_post_meta($product_id, '_dataalign_preserved_attributes', true);
        $current_da = \get_post_meta($product_id, '_dataalign_attributes', true);
        if (!is_array($current_da)) { $current_da = []; }
        $changed = false;

        if (is_array($baseline_tags) && !empty($baseline_tags)) {
            $ids = array_values(array_unique(array_map('intval', $baseline_tags)));
            \wp_set_object_terms($product_id, $ids, 'product_tag', false);
            $changed = true;
        }
        if (is_array($baseline_attrs) && !empty($baseline_attrs)) {
            // Ensure string values
            $normalized = [];
            foreach ($baseline_attrs as $k => $v) {
                $name = trim((string) $k);
                if ($name === '') { continue; }
                $val = is_array($v) ? implode(', ', array_filter(array_map('strval', $v))) : (string) $v;
                $val = trim($val);
                if ($val === '') { continue; }
                $normalized[$name] = $val;
            }
            \update_post_meta($product_id, '_dataalign_attributes', $normalized);
            $changed = true;
        } else {
            // No plugin baseline existed pre‑AI; wipe plugin attributes to remove AI-only entries if present
            if (!empty($current_da)) {
                \update_post_meta($product_id, '_dataalign_attributes', []);
                $changed = true;
            }
        }

        // Restore Woo custom attributes
        $wc_baseline = \get_post_meta($product_id, '_dataalign_preserved_wc_attributes', true);
        if (is_array($wc_baseline)) {
            \update_post_meta($product_id, '_product_attributes', $wc_baseline);
            $changed = true;
        } else {
            // Strong fallback: if baseline missing, remove AI-added custom attributes
            // Preserve taxonomy attributes and any attributes used for variations
            $wc_current = \get_post_meta($product_id, '_product_attributes', true);
            if (is_array($wc_current) && !empty($wc_current)) {
                $new_wc = [];
                foreach ($wc_current as $k => $entry) {
                    $is_tax = isset($entry['is_taxonomy']) ? (int) $entry['is_taxonomy'] : 0;
                    $is_var = isset($entry['is_variation']) ? (int) $entry['is_variation'] : 0;
                    if ($is_tax === 1 || $is_var === 1) {
                        $new_wc[$k] = $entry; // keep taxonomy attributes
                    }
                }
                if (count($new_wc) !== count($wc_current)) {
                    \update_post_meta($product_id, '_product_attributes', $new_wc);
                    $changed = true;
                }
            }
        }

        // Restore SEO baseline if present
        $pres_seo_title = (string) \get_post_meta($product_id, '_dataalign_preserved_seo_title', true);
        $pres_seo_desc  = (string) \get_post_meta($product_id, '_dataalign_preserved_seo_description', true);
        if ($pres_seo_title !== '' || $pres_seo_desc !== '') {
            // Update DataAlign plugin SEO metas
            if ($pres_seo_title !== '') { \update_post_meta($product_id, '_dataalign_seo_title', $pres_seo_title); }
            if ($pres_seo_desc !== '') { \update_post_meta($product_id, '_dataalign_seo_meta_description', $pres_seo_desc); }
            // Push into detected SEO plugin, overwriting values
            $seo_plugin = self::detect_seo_plugin($product_id);
            $limits     = self::get_seo_limits($seo_plugin['slug'] ?? 'default');
            self::map_seo_to_plugins($product_id, $pres_seo_title, $pres_seo_desc, false /* overwrite */, (string) ($seo_plugin['slug'] ?? 'none'), $limits);
            $changed = true;
        }

        // Bust Woo product caches to reflect attribute changes immediately
        if (function_exists('wc_delete_product_transients')) { \wc_delete_product_transients($product_id); }

        if (!$changed) {
            \wp_send_json_error(['message' => \esc_html__('No baseline found to restore.', 'ai-product-attributes-for-woocommerce')]);
        }

        \wp_send_json_success(['message' => \esc_html__('Restored pre‑AI tags, attributes, and SEO.', 'ai-product-attributes-for-woocommerce')]);
    }

    /**
     * AJAX: Restore baseline attributes for all products in batches.
     * Params: paged (1+), per_page (<=200)
     */
    public static function ajax_restore_all_attributes(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }

        $paged = isset($_POST['paged']) ? max(1, (int) $_POST['paged']) : 1;
        $per_page = isset($_POST['per_page']) ? (int) $_POST['per_page'] : 50;
        if ($per_page < 1) { $per_page = 50; }
        if ($per_page > 200) { $per_page = 200; }

        $q = new \WP_Query([
            'post_type'      => 'product',
            'post_status'    => 'any',
            'posts_per_page' => $per_page,
            'paged'          => $paged,
            'fields'         => 'ids',
            'no_found_rows'  => true,
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Filtering by baseline meta presence; values are indexed
            'meta_query'     => [
                'relation' => 'OR',
                [ 'key' => '_dataalign_preserved_attributes',     'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_tag_ids',         'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_seo_title',       'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_seo_description', 'compare' => 'EXISTS' ],
            ],
        ]);

        $restored = 0;
        $processed = 0;
        if ($q->have_posts()) {
            foreach ($q->posts as $pid) {
                $pid = (int) $pid;
                $processed++;
                $baseline_attrs = \get_post_meta($pid, '_dataalign_preserved_attributes', true);
                $current_da = \get_post_meta($pid, '_dataalign_attributes', true);
                if (!is_array($current_da)) { $current_da = []; }
                if (is_array($baseline_attrs) && !empty($baseline_attrs)) {
                    $normalized = [];
                    foreach ($baseline_attrs as $k => $v) {
                        $name = trim((string) $k);
                        if ($name === '') { continue; }
                        $val = is_array($v) ? implode(', ', array_filter(array_map('strval', $v))) : (string) $v;
                        $val = trim($val);
                        if ($val === '') { continue; }
                        $normalized[$name] = $val;
                    }
                    if (!empty($normalized)) {
                        \update_post_meta($pid, '_dataalign_attributes', $normalized);
                        $restored++;
                        $wc_baseline = \get_post_meta($pid, '_dataalign_preserved_wc_attributes', true);
                        if (is_array($wc_baseline)) {
                            \update_post_meta($pid, '_product_attributes', $wc_baseline);
                        } else {
                            // Strong fallback: remove AI-added custom attributes
                            // Preserve taxonomy attrs and any attributes used for variations
                            $wc_current = \get_post_meta($pid, '_product_attributes', true);
                            if (is_array($wc_current) && !empty($wc_current)) {
                                $new_wc = [];
                                foreach ($wc_current as $k => $entry) {
                                    $is_tax = isset($entry['is_taxonomy']) ? (int) $entry['is_taxonomy'] : 0;
                                    $is_var = isset($entry['is_variation']) ? (int) $entry['is_variation'] : 0;
                                    if ($is_tax === 1 || $is_var === 1) { $new_wc[$k] = $entry; }
                                }
                                \update_post_meta($pid, '_product_attributes', $new_wc);
                            }
                        }
                        if (function_exists('wc_delete_product_transients')) { \wc_delete_product_transients($pid); }
                    }
                } else {
                    // No plugin baseline; wipe plugin attrs if any
                    if (!empty($current_da)) {
                        \update_post_meta($pid, '_dataalign_attributes', []);
                        $restored++;
                    }
                    // Try to remove plugin-synced keys from Woo attrs as fallback
                    $wc_current = \get_post_meta($pid, '_product_attributes', true);
                    if (is_array($wc_current) && !empty($wc_current) && !empty($current_da)) {
                        $removed = false;
                        foreach (array_keys($current_da) as $akn) {
                            $attr_key = 'attribute_' . sanitize_title((string) $akn);
                            if (isset($wc_current[$attr_key])) {
                                // Do not remove if used for variations
                                $is_var = isset($wc_current[$attr_key]['is_variation']) ? (int) $wc_current[$attr_key]['is_variation'] : 0;
                                if ($is_var === 1) { continue; }
                                unset($wc_current[$attr_key]);
                                $removed = true;
                            }
                        }
                        if ($removed) {
                            \update_post_meta($pid, '_product_attributes', $wc_current);
                            $restored++;
                        }
                    }
                }
        // Restore tags baseline if present
        $baseline_tags = \get_post_meta($pid, '_dataalign_preserved_tag_ids', true);
        if (is_array($baseline_tags) && !empty($baseline_tags)) {
            $ids = array_values(array_unique(array_map('intval', $baseline_tags)));
            if (!empty($ids)) {
                \wp_set_object_terms($pid, $ids, 'product_tag', false);
                // Increase restored count only if attrs baseline wasn't applied
                if (!(is_array($baseline_attrs) && !empty($baseline_attrs))) {
                    $restored++;
                }
            }
        }

        // Restore SEO baseline if present
        $ps_t = (string) \get_post_meta($pid, '_dataalign_preserved_seo_title', true);
        $ps_d = (string) \get_post_meta($pid, '_dataalign_preserved_seo_description', true);
        if ($ps_t !== '' || $ps_d !== '') {
            if ($ps_t !== '') { \update_post_meta($pid, '_dataalign_seo_title', $ps_t); }
            if ($ps_d !== '') { \update_post_meta($pid, '_dataalign_seo_meta_description', $ps_d); }
            $seo_plugin = self::detect_seo_plugin($pid);
            $limits     = self::get_seo_limits($seo_plugin['slug'] ?? 'default');
            self::map_seo_to_plugins($pid, $ps_t, $ps_d, false /* overwrite */, (string) ($seo_plugin['slug'] ?? 'none'), $limits);
            $restored++;
        }
            }
        }

        $total = isset($q->found_posts) ? (int) $q->found_posts : ($processed);
        $done = ($paged * $per_page >= $total) || empty($q->posts);

        \wp_send_json_success([
            'restored'  => $restored,
            'processed' => $processed,
            'total'     => $total,
            'paged'     => $paged,
            'per_page'  => $per_page,
            'done'      => $done,
            'message'   => $done ? \esc_html__('Restore complete.', 'ai-product-attributes-for-woocommerce') : \esc_html__('Continuing…', 'ai-product-attributes-for-woocommerce'),
        ]);
    }

    /**
     * Start background restore job for baseline attributes (all products).
     */
    public static function ajax_start_restore_all_attributes(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        // Count total products with baseline
        $q = new \WP_Query([
            'post_type'      => 'product',
            'post_status'    => 'any',
            'posts_per_page' => 1,
            'paged'          => 1,
            'fields'         => 'ids',
            'no_found_rows'  => false,
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Counting products with any baseline meta present
            'meta_query'     => [
                'relation' => 'OR',
                [ 'key' => '_dataalign_preserved_attributes',     'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_tag_ids',         'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_seo_title',       'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_seo_description', 'compare' => 'EXISTS' ],
            ],
        ]);
        $total = isset($q->found_posts) ? (int) $q->found_posts : 0;
        $per_page = 50;
        $state = [
            'status'     => 'queued',
            'total'      => $total,
            'processed'  => 0,
            'restored'   => 0,
            'per_page'   => $per_page,
            'started_at' => time(),
            'updated_at' => time(),
        ];
        \update_option('_dataalign_restore_all_job', $state, false);

        // Enqueue first tick
        if (function_exists('as_enqueue_async_action')) {
            \as_enqueue_async_action('dataalign_restore_all_attributes_tick', [], 'dataalign');
        } else {
            \wp_schedule_single_event(time() + 5, 'dataalign_restore_all_attributes_tick');
        }

        \wp_send_json_success(['message' => \esc_html__('Restore queued.', 'ai-product-attributes-for-woocommerce'), 'total' => $total]);
    }

    /**
     * Get background restore job status.
     */
    public static function ajax_get_restore_all_status(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        $state = \get_option('_dataalign_restore_all_job', []);
        if (!is_array($state)) { $state = []; }
        $total = isset($state['total']) ? (int) $state['total'] : 0;
        $processed = isset($state['processed']) ? (int) $state['processed'] : 0;
        $restored = isset($state['restored']) ? (int) $state['restored'] : 0;
        $state['queued'] = $processed;
        $state['remaining'] = max(0, $total - $restored);
        $state['now'] = time();
        \wp_send_json_success($state);
    }

    /**
     * Start background bulk optimization (enrich all products) job.
     */
    public static function ajax_start_bulk_enrich(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }

        // Count total published products and initialize jobs table
        global $wpdb; $table = \esc_sql(self::jobs_table());
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $total = (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(ID) FROM {$wpdb->posts} WHERE post_type=%s AND post_status=%s", 'product', 'publish'));
        // Interactive queuing: process batches while the Optimize tab is open via AJAX polling
        $per_page = 50; // queue 50 products per interactive tick
        $run_id = uniqid('da_run_', true);
        $state = [
            'status'     => 'queued',
            'total'      => $total,
            'processed'  => 0, // number of products queued
            'applied'    => 0, // number of products enriched/applied
            'per_page'   => $per_page,
            'page'       => 1,
            'run_id'     => $run_id,
            'started_at' => time(),
            'updated_at' => time(),
            'interactive_queue' => true, // do not schedule background queue ticks; rely on Optimize tab being open
        ];
        \update_option('_dataalign_bulk_enrich_job', $state, false);

        // Prepare jobs table: clear and populate READY rows
        self::maybe_create_jobs_table();
        // Delete all existing rows for a fresh run (table name is plugin-controlled and sanitized)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query("DELETE FROM `{$table}`");
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $ids = $wpdb->get_col($wpdb->prepare("SELECT ID FROM {$wpdb->posts} WHERE post_type=%s AND post_status=%s ORDER BY ID ASC", 'product', 'publish'));
        if (!empty($ids)) {
            $now = current_time('mysql');
            foreach (array_chunk(array_map('intval', $ids), 500) as $chunk) {
                if (empty($chunk)) { continue; }
                $values = [];
                foreach ($chunk as $pid) {
                    $values[] = $wpdb->prepare("(%s,%d,%s,%s,%s)", $run_id, $pid, 'ready', $now, $now);
                }
                // Insert chunk of prepared tuples. Table name is plugin-controlled and sanitized.
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $wpdb->query("INSERT INTO `{$table}` (run_id, product_id, status, created_at, updated_at) VALUES " . implode(',', $values));
            }
        }
        // Log
        self::log('Initialized run ' . $run_id . ' with ' . (is_array($ids)?count($ids):0) . ' READY');

        \wp_send_json_success(['message' => \esc_html__('Bulk optimization queued.', 'ai-product-attributes-for-woocommerce'), 'total' => $total]);
    }

    /**
     * Get bulk optimization job status.
     */
    public static function ajax_get_bulk_enrich_status(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        $state = \get_option('_dataalign_bulk_enrich_job', []);
        if (!is_array($state)) { $state = []; }
        $run_id = isset($state['run_id']) ? (string) $state['run_id'] : '';
        $total = 0; $applied = 0; $processed = 0;
        if ($run_id !== '') {
            global $wpdb; $table = \esc_sql(self::jobs_table());
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $row = $wpdb->get_row(
                $wpdb->prepare(
                    /* phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared */
                    "SELECT COUNT(*) AS total, "
                    . "SUM(status IN ('queued','completed')) AS processed, "
                    . "SUM(status='completed') AS applied FROM `{$table}` WHERE run_id=%s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $run_id
                ),
                ARRAY_A
            );
            if (is_array($row)) {
                $total = (int) ($row['total'] ?? 0);
                $processed = (int) ($row['processed'] ?? 0);
                $applied = (int) ($row['applied'] ?? 0);
            }
        }
        $remaining = max(0, $total - $applied);
        // Derive status: completed only when applied reaches total
        $status = isset($state['status']) ? (string) $state['status'] : 'queued';
        // Consider completed when all queued have been applied, or total reached
        if ($status !== 'cancelled') {
            if ($total > 0 && $applied >= $total) {
                $status = 'completed';
            } elseif ($processed > 0 && $applied >= $processed) {
                $status = 'completed';
            }
        }
        $state['now'] = time();
        $state['remaining'] = $remaining;
        $state['queued'] = $processed;
        $state['applied'] = $applied;
        $state['status'] = $status;
        // expose plan limit and upgrade url if present
        $state['planLimit'] = isset($state['planLimit']) ? (bool) $state['planLimit'] : false;
        if ($state['planLimit']) {
            $state['upgradeUrl'] = isset($state['upgradeUrl']) ? (string) $state['upgradeUrl'] : (defined(__NAMESPACE__ . '\\UPGRADE_URL') ? constant(__NAMESPACE__ . '\\UPGRADE_URL') : 'https://dataalign.ai/pricing');
        }
        \wp_send_json_success($state);
    }

    /**
     * AJAX: Poll job statuses immediately (on-demand) and return updated status.
     */
    public static function ajax_poll_now(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        // Advance QUEUE only (interactive queuing) — no apply here
        try { self::process_queue_from_table(50); } catch (\Throwable $e) {}
        // Return latest status from jobs table only
        $state = \get_option('_dataalign_bulk_enrich_job', []);
        if (!is_array($state)) { $state = []; }
        $run_id = isset($state['run_id']) ? (string) $state['run_id'] : '';
        $total = 0; $applied = 0; $processed = 0;
        if ($run_id !== '') {
            global $wpdb; $table = \esc_sql(self::jobs_table());
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $row = $wpdb->get_row(
                $wpdb->prepare(
                    /* phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared */
                    "SELECT COUNT(*) AS total, "
                    . "SUM(status IN ('queued','completed')) AS processed, "
                    . "SUM(status='completed') AS applied FROM `{$table}` WHERE run_id=%s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $run_id
                ),
                ARRAY_A
            );
            if (is_array($row)) {
                $total = (int) ($row['total'] ?? 0);
                $processed = (int) ($row['processed'] ?? 0);
                $applied = (int) ($row['applied'] ?? 0);
            }
        }
        $remaining = max(0, $total - $applied);
        // Apply phase handled by cron to avoid heavy work during UI polling
        $status = isset($state['status']) ? (string) $state['status'] : 'queued';
        if ($status !== 'cancelled') {
            if ($total > 0 && $applied >= $total) {
                $status = 'completed';
            } elseif ($processed > 0 && $applied >= $processed) {
                $status = 'completed';
            }
        }
        $state['now'] = time();
        $state['remaining'] = $remaining;
        $state['queued'] = $processed;
        $state['applied'] = $applied;
        $state['status'] = $status;
        $state['planLimit'] = isset($state['planLimit']) ? (bool) $state['planLimit'] : false;
        if ($state['planLimit']) {
            $state['upgradeUrl'] = isset($state['upgradeUrl']) ? (string) $state['upgradeUrl'] : (defined(__NAMESPACE__ . '\\UPGRADE_URL') ? constant(__NAMESPACE__ . '\\UPGRADE_URL') : 'https://dataalign.ai/pricing');
        }
        // attach logs
        $state['logs'] = get_option('_dataalign_logs', []);
        \wp_send_json_success($state);
    }

    /**
     * Cancel current bulk optimization job.
     */
    public static function ajax_cancel_bulk_enrich(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';
        if (!\wp_verify_nonce($nonce, 'dataalign_settings')) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('manage_woocommerce')) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }

        $state = \get_option('_dataalign_bulk_enrich_job', []);
        if (!is_array($state) || empty($state)) {
            \wp_send_json_error(['message' => \esc_html__('No bulk job to cancel.', 'ai-product-attributes-for-woocommerce')]);
        }
        $state['status'] = 'cancelled';
        $state['updated_at'] = time();
        \update_option('_dataalign_bulk_enrich_job', $state, false);
        \wp_send_json_success(['message' => \esc_html__('Bulk optimization cancelled.', 'ai-product-attributes-for-woocommerce')]);
    }

    /**
     * Process a background tick of the bulk optimization job.
     */
    public static function process_bulk_enrich_tick(): void
    {
        $state = \get_option('_dataalign_bulk_enrich_job', []);
        if (!is_array($state) || empty($state)) { return; }
        $status = isset($state['status']) ? (string) $state['status'] : 'queued';
        $total = isset($state['total']) ? (int) $state['total'] : 0;
        $processed = isset($state['processed']) ? (int) $state['processed'] : 0;
        $per_page = isset($state['per_page']) ? (int) $state['per_page'] : 25;
        $page = isset($state['page']) ? (int) $state['page'] : 1;
        if ($status === 'cancelled') {
            $state['updated_at'] = time();
            \update_option('_dataalign_bulk_enrich_job', $state, false);
            return;
        }
        if ($status === 'completed' || $total === 0 || $processed >= $total) {
            $state['status'] = 'completed';
            $state['updated_at'] = time();
            \update_option('_dataalign_bulk_enrich_job', $state, false);
            return;
        }
        $state['status'] = 'running';
        $targets = isset($state['targets']) && is_array($state['targets']) ? array_values(array_map('intval', $state['targets'])) : [];
        $batch_processed = 0;
        $products_payload = [];
        if (!empty($targets)) {
            // Gather existing store tags once per request for reuse in each product payload
            $existing_tags = self::get_all_product_tag_names();
            $offset = max(0, ($page - 1) * max(1, $per_page));
            $batch_ids = array_slice($targets, $offset, $per_page);
            foreach ($batch_ids as $pid) {
                $pid = (int) $pid;
                $post = \get_post($pid);
                if (!$post || $post->post_type !== 'product') { continue; }
                $seo_plugin = self::detect_seo_plugin($pid);
                $seo_limits = self::get_seo_limits($seo_plugin['slug'] ?? 'default');
                // Build images including variation images unless excluded
                $images = [];
                $exclude_images = (bool) \get_option('dataalign_exclude_images', false);
                if (!$exclude_images && function_exists('wc_get_product')) {
                    $pobj = \wc_get_product($pid);
                    if ($pobj) {
                        $thumb_id = (int) \get_post_thumbnail_id($pid);
                        if ($thumb_id) { $u = \wp_get_attachment_image_url($thumb_id, 'full'); if ($u) { $images[] = $u; } }
                        $gallery_ids = method_exists($pobj, 'get_gallery_image_ids') ? (array) $pobj->get_gallery_image_ids() : [];
                        foreach ($gallery_ids as $img_id) {
                            $u = \wp_get_attachment_image_url((int) $img_id, 'full'); if ($u) { $images[] = $u; }
                        }
                        if (method_exists($pobj, 'is_type') && $pobj->is_type('variable')) {
                            $variation_ids = method_exists($pobj, 'get_children') ? (array) $pobj->get_children() : [];
                            foreach ($variation_ids as $vid) {
                                $v = \wc_get_product((int) $vid);
                                if ($v && method_exists($v, 'get_image_id')) {
                                    $v_img_id = (int) $v->get_image_id();
                                    if ($v_img_id) { $u = \wp_get_attachment_image_url($v_img_id, 'full'); if ($u) { $images[] = $u; } }
                                }
                            }
                        }
                    }
                    $public_base = (string) \get_option('dataalign_public_base_url', '');
                    $site_base   = \home_url();
                    if ($public_base !== '' && $site_base) {
                        $public_base = rtrim($public_base, '/');
                        $site_base   = rtrim($site_base, '/');
                        $images = array_map(function ($u) use ($public_base, $site_base) { return is_string($u) ? str_replace($site_base, $public_base, $u) : $u; }, $images);
                    }
                    $images = array_values(array_unique(array_filter(array_map('strval', $images))));
                }
                $products_payload[] = [
                    'product_id'  => (string) $pid,
                    'title'       => (string) $post->post_title,
                    'description' => (string) $post->post_content,
                    'images'      => $images,
                    'existing_tags'   => $existing_tags,
                    'seo_plugin'       => (string) ($seo_plugin['slug'] ?? 'none'),
                    'seo_plugin_name'  => (string) ($seo_plugin['name'] ?? 'None'),
                    'seo_limits'       => [
                        'title_max'       => (int) ($seo_limits['title_max'] ?? 60),
                        'description_max' => (int) ($seo_limits['description_max'] ?? 160),
                    ],
                ];
            }
            if (!empty($products_payload)) {
                $client = new DataAlign_API_Client();
                $resp = $client->enrich_products_bulk($products_payload);
                if (!\is_wp_error($resp)) {
                    $success = isset($resp['success']) ? (bool) $resp['success'] : false;
                    $data    = isset($resp['data']) && is_array($resp['data']) ? $resp['data'] : [];
                    $job_ids = isset($data['job_ids']) && is_array($data['job_ids']) ? $data['job_ids'] : [];
                    $count = min(count($job_ids), count($products_payload));
                    for ($i = 0; $i < $count; $i++) {
                        $pid = (int) $products_payload[$i]['product_id'];
                        $job_id = (int) $job_ids[$i];
                        $all_jobs = (array) \get_post_meta($pid, '_dataalign_job_ids', true);
                        $all_jobs[] = $job_id;
                        $all_jobs = array_values(array_unique(array_map('intval', $all_jobs)));
                        \update_post_meta($pid, '_dataalign_job_ids', $all_jobs);
                        \update_post_meta($pid, '_dataalign_has_pending_jobs', true);
                        \update_option('_dataalign_any_pending', 1, false);
                        if (!empty($state['run_id'])) { \update_post_meta($pid, '_dataalign_bulk_run', (string) $state['run_id']); }
                    }
                } else {
                    if ($resp->get_error_code() === 'dataalign_plan_limit') {
                        $state['status'] = 'cancelled';
                        $state['planLimit'] = true;
                        $state['upgradeUrl'] = defined(__NAMESPACE__ . '\\UPGRADE_URL') ? constant(__NAMESPACE__ . '\\UPGRADE_URL') : 'https://dataalign.ai/pricing';
                        $state['updated_at'] = time();
                        \update_option('_dataalign_bulk_enrich_job', $state, false);
                        return;
                    }
                    \update_option('_dataalign_last_global_error', $resp->get_error_message());
                }
            }
            $batch_processed = count($products_payload);
        } else {
            // Fallback to legacy: process all products when no explicit targets set
            $q = new \WP_Query([
                'post_type'      => 'product',
                'post_status'    => 'any',
                'posts_per_page' => $per_page,
                'paged'          => max(1, $page),
                'fields'         => 'ids',
                'orderby'        => 'ID',
                'order'          => 'ASC',
                'no_found_rows'  => true,
            ]);
            if ($q->have_posts()) {
                // Gather existing store tags once per request for reuse in each product payload
                $existing_tags = self::get_all_product_tag_names();
                foreach ($q->posts as $pid) {
                    $pid = (int) $pid;
                    $post = \get_post($pid);
                    if (!$post || $post->post_type !== 'product') { continue; }
                    $seo_plugin = self::detect_seo_plugin($pid);
                    $seo_limits = self::get_seo_limits($seo_plugin['slug'] ?? 'default');
                    // Build images including variation images unless excluded
                    $images = [];
                    $exclude_images = (bool) \get_option('dataalign_exclude_images', false);
                    if (!$exclude_images && function_exists('wc_get_product')) {
                        $pobj = \wc_get_product($pid);
                        if ($pobj) {
                            $thumb_id = (int) \get_post_thumbnail_id($pid);
                            if ($thumb_id) { $u = \wp_get_attachment_image_url($thumb_id, 'full'); if ($u) { $images[] = $u; } }
                            $gallery_ids = method_exists($pobj, 'get_gallery_image_ids') ? (array) $pobj->get_gallery_image_ids() : [];
                            foreach ($gallery_ids as $img_id) {
                                $u = \wp_get_attachment_image_url((int) $img_id, 'full'); if ($u) { $images[] = $u; }
                            }
                            if (method_exists($pobj, 'is_type') && $pobj->is_type('variable')) {
                                $variation_ids = method_exists($pobj, 'get_children') ? (array) $pobj->get_children() : [];
                                foreach ($variation_ids as $vid) {
                                    $v = \wc_get_product((int) $vid);
                                    if ($v && method_exists($v, 'get_image_id')) {
                                        $v_img_id = (int) $v->get_image_id();
                                        if ($v_img_id) { $u = \wp_get_attachment_image_url($v_img_id, 'full'); if ($u) { $images[] = $u; } }
                                    }
                                }
                            }
                        }
                        $public_base = (string) \get_option('dataalign_public_base_url', '');
                        $site_base   = \home_url();
                        if ($public_base !== '' && $site_base) {
                            $public_base = rtrim($public_base, '/');
                            $site_base   = rtrim($site_base, '/');
                            $images = array_map(function ($u) use ($public_base, $site_base) { return is_string($u) ? str_replace($site_base, $public_base, $u) : $u; }, $images);
                        }
                        $images = array_values(array_unique(array_filter(array_map('strval', $images))));
                    }
                    $products_payload[] = [
                        'product_id'  => (string) $pid,
                        'title'       => (string) $post->post_title,
                        'description' => (string) $post->post_content,
                        'images'      => $images,
                        'existing_tags'   => $existing_tags,
                        'seo_plugin'       => (string) ($seo_plugin['slug'] ?? 'none'),
                        'seo_plugin_name'  => (string) ($seo_plugin['name'] ?? 'None'),
                        'seo_limits'       => [
                            'title_max'       => (int) ($seo_limits['title_max'] ?? 60),
                            'description_max' => (int) ($seo_limits['description_max'] ?? 160),
                        ],
                    ];
                }
                if (!empty($products_payload)) {
                    $client = new DataAlign_API_Client();
                    $resp = $client->enrich_products_bulk($products_payload);
                    if (!\is_wp_error($resp)) {
                        $success = isset($resp['success']) ? (bool) $resp['success'] : false;
                        $data    = isset($resp['data']) && is_array($resp['data']) ? $resp['data'] : [];
                        // Fast-path: if API returns results for the batch, apply them immediately
                        $results = isset($data['results']) && is_array($data['results']) ? $data['results'] : [];
                        if (!empty($results)) {
                            foreach ($results as $res) {
                                $pid = isset($res['product_id']) ? (int) $res['product_id'] : 0;
                                if ($pid <= 0) { continue; }
                                $payload = [
                                    'tags'       => isset($res['tags']) && is_array($res['tags']) ? $res['tags'] : [],
                                    'attributes' => isset($res['attributes']) && is_array($res['attributes']) ? $res['attributes'] : [],
                                    'seo'        => isset($res['seo']) && is_array($res['seo']) ? $res['seo'] : [],
                                    'job_id'     => isset($res['job_id']) ? (int) $res['job_id'] : 0,
                                ];
                                self::apply_enrichment_data($pid, $payload, $res);
                                if (!empty($state['run_id'])) { \update_post_meta($pid, '_dataalign_bulk_run', (string) $state['run_id']); }
                                $p_applied = (string) \get_post_meta($pid, '_dataalign_bulk_applied', true);
                                if ($p_applied !== (string) ($state['run_id'] ?? '')) {
                                    $applied = isset($state['applied']) ? (int) $state['applied'] : 0;
                                    $state['applied'] = (int) ($applied + 1);
                                    \update_post_meta($pid, '_dataalign_bulk_applied', (string) ($state['run_id'] ?? ''));
                                }
                                \update_post_meta($pid, '_dataalign_has_pending_jobs', false);
                            }
                        }
                        // Capture job_ids for any still-running items
                        $job_ids = isset($data['job_ids']) && is_array($data['job_ids']) ? $data['job_ids'] : [];
                        $count = min(count($job_ids), count($products_payload));
                        for ($i = 0; $i < $count; $i++) {
                            $pid = (int) $products_payload[$i]['product_id'];
                            $job_id = (int) $job_ids[$i];
                            $all_jobs = (array) \get_post_meta($pid, '_dataalign_job_ids', true);
                            $all_jobs[] = $job_id;
                            $all_jobs = array_values(array_unique(array_map('intval', $all_jobs)));
                            \update_post_meta($pid, '_dataalign_job_ids', $all_jobs);
                            \update_post_meta($pid, '_dataalign_has_pending_jobs', true);
                            \update_option('_dataalign_any_pending', 1, false);
                            if (!empty($state['run_id'])) { \update_post_meta($pid, '_dataalign_bulk_run', (string) $state['run_id']); }
                        }
                        // Immediately try a bulk status call to convert any just-queued jobs to applied results in one go
                        if (empty($results) && !empty($job_ids)) {
                            $multi = $client->get_jobs_status($job_ids);
                            if (!\is_wp_error($multi) && isset($multi['data']['jobs']) && is_array($multi['data']['jobs'])) {
                                foreach ($multi['data']['jobs'] as $j) {
                                    $jid = isset($j['id']) ? (int) $j['id'] : 0;
                                    $status = isset($j['status']) ? (string) $j['status'] : '';
                                    $result = isset($j['result']) && is_array($j['result']) ? $j['result'] : null;
                                    if ($jid <= 0 || $status !== 'completed' || empty($result)) { continue; }
                                    $idx = array_search($jid, $job_ids, true);
                                    if ($idx === false) { continue; }
                                    $pid = (int) $products_payload[$idx]['product_id'];
                                    self::apply_enrichment_data($pid, $result, is_array($j) ? $j : []);
                                    if (!empty($state['run_id'])) { \update_post_meta($pid, '_dataalign_bulk_run', (string) $state['run_id']); }
                                    $p_applied = (string) \get_post_meta($pid, '_dataalign_bulk_applied', true);
                                    if ($p_applied !== (string) ($state['run_id'] ?? '')) {
                                        $applied = isset($state['applied']) ? (int) $state['applied'] : 0;
                                        $state['applied'] = (int) ($applied + 1);
                                        \update_post_meta($pid, '_dataalign_bulk_applied', (string) ($state['run_id'] ?? ''));
                                    }
                                    \update_post_meta($pid, '_dataalign_has_pending_jobs', false);
                                    $completed_jobs = (array) \get_post_meta($pid, '_dataalign_completed_job_ids', true);
                                    $completed_jobs[] = $jid;
                                    $completed_jobs = array_values(array_unique(array_map('intval', $completed_jobs)));
                                    \update_post_meta($pid, '_dataalign_completed_job_ids', $completed_jobs);
                                }
                            }
                        }
                    } else {
                        if ($resp->get_error_code() === 'dataalign_plan_limit') {
                            $state['status'] = 'cancelled';
                            $state['planLimit'] = true;
                            $state['upgradeUrl'] = defined(__NAMESPACE__ . '\\UPGRADE_URL') ? constant(__NAMESPACE__ . '\\UPGRADE_URL') : 'https://dataalign.ai/pricing';
                            $state['updated_at'] = time();
                            \update_option('_dataalign_bulk_enrich_job', $state, false);
                            return;
                        }
                        \update_option('_dataalign_last_global_error', $resp->get_error_message());
                    }
                }
                $batch_processed = count($q->posts);
            }
        }

        $state['processed'] = (int) min($total, $processed + $batch_processed);
        $state['page'] = (int) ($page + 1);
        $state['updated_at'] = time();
        // If queuing is done (all processed or no posts), do not enqueue next tick; wait for apply phase.
        $more = ($state['processed'] < $total);
        // In interactive queue mode, do NOT schedule background ticks; the Optimize tab polling will drive queuing
        $interactive = !empty($state['interactive_queue']);
        if ($more && !$interactive) {
            if (function_exists('as_enqueue_async_action')) {
                \as_enqueue_async_action('dataalign_bulk_enrich_tick', [], 'dataalign');
            } else {
                \wp_schedule_single_event(time() + 2, 'dataalign_bulk_enrich_tick');
            }
        }
        \update_option('_dataalign_bulk_enrich_job', $state, false);
    }

    /**
     * Take a batch of READY products from jobs table, send to bulk enrich API, and mark QUEUED with job_id.
     */
    protected static function process_queue_from_table(int $batch_size = 50): void
    {
        $state = \get_option('_dataalign_bulk_enrich_job', []);
        if (!is_array($state) || empty($state)) { return; }
        $run_id = isset($state['run_id']) ? (string) $state['run_id'] : '';
        if ($run_id === '') { return; }
        global $wpdb; $table = \esc_sql(self::jobs_table());
        // Fetch READY rows
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $rows = $wpdb->get_results(
            /* phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared */
            $wpdb->prepare(
                "SELECT product_id FROM `{$table}` WHERE run_id=%s AND status=%s ORDER BY product_id ASC LIMIT %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $run_id,
                'ready',
                max(1,(int)$batch_size)
            ),
            ARRAY_A
        );
        if (empty($rows)) { return; }
        $products_payload = [];
        // Gather existing store tags once per request for reuse in each product payload
        $existing_tags = self::get_all_product_tag_names();
        foreach ($rows as $r) {
            $pid = (int) $r['product_id'];
            $post = \get_post($pid);
            if (!$post || $post->post_type !== 'product') { continue; }
            $seo_plugin = self::detect_seo_plugin($pid);
            $seo_limits = self::get_seo_limits($seo_plugin['slug'] ?? 'default');
            // Build images including variation images unless excluded
            $images = [];
            $exclude_images = (bool) \get_option('dataalign_exclude_images', false);
            if (!$exclude_images && function_exists('wc_get_product')) {
                $pobj = \wc_get_product($pid);
                if ($pobj) {
                    $thumb_id = (int) \get_post_thumbnail_id($pid);
                    if ($thumb_id) { $u = \wp_get_attachment_image_url($thumb_id, 'full'); if ($u) { $images[] = $u; } }
                    $gallery_ids = method_exists($pobj, 'get_gallery_image_ids') ? (array) $pobj->get_gallery_image_ids() : [];
                    foreach ($gallery_ids as $img_id) {
                        $u = \wp_get_attachment_image_url((int) $img_id, 'full'); if ($u) { $images[] = $u; }
                    }
                    if (method_exists($pobj, 'is_type') && $pobj->is_type('variable')) {
                        $variation_ids = method_exists($pobj, 'get_children') ? (array) $pobj->get_children() : [];
                        foreach ($variation_ids as $vid) {
                            $v = \wc_get_product((int) $vid);
                            if ($v && method_exists($v, 'get_image_id')) {
                                $v_img_id = (int) $v->get_image_id();
                                if ($v_img_id) { $u = \wp_get_attachment_image_url($v_img_id, 'full'); if ($u) { $images[] = $u; } }
                            }
                        }
                    }
                }
                $public_base = (string) \get_option('dataalign_public_base_url', '');
                $site_base   = \home_url();
                if ($public_base !== '' && $site_base) {
                    $public_base = rtrim($public_base, '/');
                    $site_base   = rtrim($site_base, '/');
                    $images = array_map(function ($u) use ($public_base, $site_base) { return is_string($u) ? str_replace($site_base, $public_base, $u) : $u; }, $images);
                }
                $images = array_values(array_unique(array_filter(array_map('strval', $images))));
            }
            $products_payload[] = [
                'product_id'  => (string) $pid,
                'title'       => (string) $post->post_title,
                'description' => (string) $post->post_content,
                'images'      => $images,
                'existing_tags'   => $existing_tags,
                'seo_plugin'       => (string) ($seo_plugin['slug'] ?? 'none'),
                'seo_plugin_name'  => (string) ($seo_plugin['name'] ?? 'None'),
                'seo_limits'       => [
                    'title_max'       => (int) ($seo_limits['title_max'] ?? 60),
                    'description_max' => (int) ($seo_limits['description_max'] ?? 160),
                ],
            ];
        }
        if (empty($products_payload)) { return; }
        $client = new DataAlign_API_Client();
        $resp = $client->enrich_products_bulk($products_payload);
        if (\is_wp_error($resp)) { self::log('Queue error: ' . $resp->get_error_message()); return; }
        $data = isset($resp['data']) && is_array($resp['data']) ? $resp['data'] : [];
        $job_ids = isset($data['job_ids']) && is_array($data['job_ids']) ? $data['job_ids'] : [];
        $count = min(count($job_ids), count($products_payload));
        for ($i = 0; $i < $count; $i++) {
            $pid = (int) $products_payload[$i]['product_id'];
            $jid = (int) $job_ids[$i];
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->update($table, [
                'status'     => 'queued',
                'job_id'     => $jid,
                'queued_at'  => current_time('mysql'),
                'updated_at' => current_time('mysql'),
            ], [ 'run_id' => $run_id, 'product_id' => $pid ], [ '%s','%d','%s','%s' ], [ '%s','%d' ]);
        }
        $state['processed'] = (int) ($state['processed'] ?? 0) + $count;
        $state['updated_at'] = time();
        \update_option('_dataalign_bulk_enrich_job', $state, false);
        self::log('Queued ' . $count . ' products');
        \update_option('_dataalign_any_pending', 1, false);
    }

    /**
     * Apply phase: read up to $max queued jobs from jobs table, fetch statuses in bulk, apply completed, mark completed.
     */
    protected static function process_apply_from_table(int $max = 50): void
    {
        $state = \get_option('_dataalign_bulk_enrich_job', []);
        if (!is_array($state) || empty($state)) { return; }
        $run_id = isset($state['run_id']) ? (string) $state['run_id'] : '';
        if ($run_id === '') { return; }
        global $wpdb; $table = \esc_sql(self::jobs_table());
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $rows = $wpdb->get_results(
            /* phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared */
            $wpdb->prepare(
                "SELECT product_id, job_id FROM `{$table}` WHERE run_id=%s AND status=%s AND job_id IS NOT NULL ORDER BY product_id ASC LIMIT %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $run_id,
                'queued',
                max(1,(int)$max)
            ),
            ARRAY_A
        );
        if (empty($rows)) { return; }
        $job_to_pid = [];
        $ids = [];
        foreach ($rows as $r) { $jid = (int) $r['job_id']; if ($jid > 0) { $job_to_pid[$jid] = (int) $r['product_id']; $ids[] = $jid; } }
        if (empty($ids)) { return; }
        $client = new DataAlign_API_Client();
        $multi = $client->get_jobs_status($ids);
        if (\is_wp_error($multi)) { self::log('Apply error: ' . $multi->get_error_message()); return; }
        $jobs = isset($multi['data']['jobs']) && is_array($multi['data']['jobs']) ? $multi['data']['jobs'] : [];
        $applied_now = 0;
        $completed_pids = [];
        $now_mysql = current_time('mysql');
        foreach ($jobs as $j) {
            $jid = isset($j['id']) ? (int) $j['id'] : 0;
            $status = isset($j['status']) ? (string) $j['status'] : '';
            $result = isset($j['result']) && is_array($j['result']) ? $j['result'] : null;
            if ($jid <= 0 || $status !== 'completed' || empty($result)) { continue; }
            if (!isset($job_to_pid[$jid])) { continue; }
            $pid = (int) $job_to_pid[$jid];
            // Apply product changes (per-product writes are required for terms/meta)
            self::apply_enrichment_data($pid, $result, is_array($j) ? $j : []);
            $completed_pids[] = $pid;
            $applied_now++;
        }
        // Single bulk UPDATE for all completed rows in this batch
        if (!empty($completed_pids)) {
            $placeholders = implode(',', array_fill(0, count($completed_pids), '%d'));
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $wpdb->query(
                // phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
                $wpdb->prepare(
                    /* phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber */
                    "UPDATE `{$table}` SET status='completed', applied_at=%s, updated_at=%s WHERE run_id=%s AND product_id IN ({$placeholders})", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
                    $now_mysql, $now_mysql, $run_id, ...$completed_pids
                )
            );
        }
        if ($applied_now > 0) { self::log('Applied ' . $applied_now . ' products'); }
        // Update state counters from table
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $row = $wpdb->get_row(
                /* phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared */
                $wpdb->prepare(
                    "SELECT COUNT(*) AS total, "
                    . "SUM(status IN ('queued','completed')) AS processed, "
                    . "SUM(status='completed') AS applied FROM `{$table}` WHERE run_id=%s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $run_id
                ),
                ARRAY_A
            );
        if (is_array($row)) {
            $state['total'] = (int) ($row['total'] ?? 0);
            $state['processed'] = (int) ($row['processed'] ?? 0);
            $state['applied'] = (int) ($row['applied'] ?? 0);
            $state['updated_at'] = time();
            if ($state['total'] > 0 && $state['applied'] >= $state['total']) { $state['status'] = 'completed'; }
            \update_option('_dataalign_bulk_enrich_job', $state, false);
        }
    }

    /**
     * Process a background tick of the restore job.
     */
    public static function process_restore_all_tick(): void
    {
        $state = \get_option('_dataalign_restore_all_job', []);
        if (!is_array($state) || empty($state)) {
            return; // nothing to do
        }
        $status = isset($state['status']) ? (string) $state['status'] : 'queued';
        $total = isset($state['total']) ? (int) $state['total'] : 0;
        $processed = isset($state['processed']) ? (int) $state['processed'] : 0;
        $restored = isset($state['restored']) ? (int) $state['restored'] : 0;
        $per_page = isset($state['per_page']) ? (int) $state['per_page'] : 50;
        if ($status === 'completed' || $total === 0 || $processed >= $total) {
            $state['status'] = 'completed';
            $state['updated_at'] = time();
            \update_option('_dataalign_restore_all_job', $state, false);
            return;
        }
        $state['status'] = 'running';

        $paged = (int) floor($processed / max(1, $per_page)) + 1;
        $q = new \WP_Query([
            'post_type'      => 'product',
            'post_status'    => 'any',
            'posts_per_page' => $per_page,
            'paged'          => $paged,
            'fields'         => 'ids',
            'no_found_rows'  => true,
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Filtering by baseline meta presence during restore
            'meta_query'     => [
                'relation' => 'OR',
                [ 'key' => '_dataalign_preserved_attributes',     'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_tag_ids',         'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_seo_title',       'compare' => 'EXISTS' ],
                [ 'key' => '_dataalign_preserved_seo_description', 'compare' => 'EXISTS' ],
            ],
        ]);
        $batch_processed = 0;
        $batch_restored = 0;
        if ($q->have_posts()) {
            foreach ($q->posts as $pid) {
                $pid = (int) $pid;
                $batch_processed++;
                $baseline_attrs = \get_post_meta($pid, '_dataalign_preserved_attributes', true);
                $current_da = \get_post_meta($pid, '_dataalign_attributes', true);
                if (!is_array($current_da)) { $current_da = []; }
                if (is_array($baseline_attrs) && !empty($baseline_attrs)) {
                    $normalized = [];
                    foreach ($baseline_attrs as $k => $v) {
                        $name = trim((string) $k);
                        if ($name === '') { continue; }
                        $val = is_array($v) ? implode(', ', array_filter(array_map('strval', $v))) : (string) $v;
                        $val = trim($val);
                        if ($val === '') { continue; }
                        $normalized[$name] = $val;
                    }
                    if (!empty($normalized)) {
                        \update_post_meta($pid, '_dataalign_attributes', $normalized);
                        $batch_restored++;
                        // Restore Woo attributes baseline exactly if present, else fallback to sync
                        $wc_baseline = \get_post_meta($pid, '_dataalign_preserved_wc_attributes', true);
                        if (is_array($wc_baseline)) {
                            \update_post_meta($pid, '_product_attributes', $wc_baseline);
                        } elseif ((bool) \get_option('dataalign_sync_attributes_to_woo', true)) {
                            self::sync_attributes_to_woo($pid, $normalized, false /* overwrite */);
                        }
                    }
                } else {
                    // No plugin baseline; wipe plugin attrs if any
                    if (!empty($current_da)) {
                        \update_post_meta($pid, '_dataalign_attributes', []);
                        $batch_restored++;
                    }
                    // Try to remove plugin-synced keys from Woo attrs as fallback
                    $wc_current = \get_post_meta($pid, '_product_attributes', true);
                    if (is_array($wc_current) && !empty($wc_current)) {
                        $new_wc = [];
                        foreach ($wc_current as $k => $entry) {
                            $is_tax = isset($entry['is_taxonomy']) ? (int) $entry['is_taxonomy'] : 0;
                            if ($is_tax === 1) { $new_wc[$k] = $entry; }
                        }
                        if (count($new_wc) !== count($wc_current)) {
                            \update_post_meta($pid, '_product_attributes', $new_wc);
                            if (function_exists('wc_delete_product_transients')) { \wc_delete_product_transients($pid); }
                            $batch_restored++;
                        }
                    }
                }
                // Restore tags baseline if present
                $baseline_tags = \get_post_meta($pid, '_dataalign_preserved_tag_ids', true);
                if (is_array($baseline_tags) && !empty($baseline_tags)) {
                    $ids = array_values(array_unique(array_map('intval', $baseline_tags)));
                    if (!empty($ids)) {
                        \wp_set_object_terms($pid, $ids, 'product_tag', false);
                        // Count as restored if tags baseline applied and attrs baseline was empty
                        if (!(is_array($baseline_attrs) && !empty($baseline_attrs))) {
                            $batch_restored++;
                        }
                    }
                }
            }
        }

        $state['processed'] = min($total, $processed + $batch_processed);
        $state['restored']  = (int) ($restored + $batch_restored);
        $state['updated_at'] = time();

        if ($state['processed'] >= $total || empty($q->posts)) {
            $state['status'] = 'completed';
        }
        \update_option('_dataalign_restore_all_job', $state, false);

        // Queue next tick if not done
        if ($state['status'] !== 'completed') {
            if (function_exists('as_enqueue_async_action')) {
                \as_enqueue_async_action('dataalign_restore_all_attributes_tick', [], 'dataalign');
            } else {
                \wp_schedule_single_event(time() + 10, 'dataalign_restore_all_attributes_tick');
            }
        }
    }

    /**
     * Render attributes as a compact table (admin view).
     *
     * @param array $attrs
     * @return void
     */
    protected static function render_attributes_table(array $attrs): void
    {
        if (empty($attrs)) {
        echo '<p>' . \esc_html__('No attributes available.', 'ai-product-attributes-for-woocommerce') . '</p>';
            return;
        }
        echo '<table class="widefat striped" style="margin-top:6px">';
        echo '<tbody>';
        foreach ($attrs as $k => $v) {
            $name = self::humanize_attribute_key((string) $k);
            $value = is_array($v) ? implode(', ', array_filter(array_map('strval', $v))) : (string) $v;
            echo '<tr><td style="width:40%"><strong>' . \esc_html($name) . '</strong></td><td>' . \esc_html($value) . '</td></tr>';
        }
        echo '</tbody>';
        echo '</table>';
    }

    /**
     * Add a product tab rendering DataAlign attributes.
     */
    public static function add_specs_tab(array $tabs): array
    {
        $enabled = (bool) \get_option('dataalign_enable_specs_tab', false);
        if (!$enabled || !is_product()) {
            return $tabs;
        }
        global $post;
        if (!$post) {
            return $tabs;
        }
        $attrs = \get_post_meta($post->ID, '_dataalign_attributes', true);
        if (!is_array($attrs) || empty($attrs)) {
            return $tabs;
        }
        $title = (string) \get_option('dataalign_specs_tab_title', __('Specifications', 'ai-product-attributes-for-woocommerce'));
        $tabs['dataalign_specs'] = [
            'title'    => \esc_html($title),
            'priority' => 25,
            'callback' => [__CLASS__, 'render_specs_tab'],
        ];
        // Optionally hide WooCommerce's Additional Information tab to avoid duplicate content
        if ((bool) \get_option('dataalign_hide_additional_info_tab', false)) {
            if (isset($tabs['additional_information'])) {
                unset($tabs['additional_information']);
            }
        }
        return $tabs;
    }

    /**
     * Render callback for product specs tab (frontend).
     */
    public static function render_specs_tab(): void
    {
        $product_id = get_the_ID();
        $attrs = \get_post_meta($product_id, '_dataalign_attributes', true);
        if (!is_array($attrs) || empty($attrs)) {
            echo '<p>' . \esc_html__('No specifications available.', 'ai-product-attributes-for-woocommerce') . '</p>';
            return;
        }
        echo '<table class="shop_attributes">';
        foreach ($attrs as $k => $v) {
            $name = self::humanize_attribute_key((string) $k);
            $value = is_array($v) ? implode(', ', array_filter(array_map('strval', $v))) : (string) $v;
            echo '<tr><th>' . \esc_html($name) . '</th><td>' . \esc_html($value) . '</td></tr>';
        }
        echo '</table>';
    }

    /**
     * Late filter to hide WooCommerce Additional information tab when opted in.
     */
    public static function maybe_hide_additional_info_tab(array $tabs): array
    {
        if (!function_exists('is_product') || !is_product()) {
            return $tabs;
        }
        if (!(bool) \get_option('dataalign_hide_additional_info_tab', false)) {
            return $tabs;
        }
        if (isset($tabs['additional_information'])) {
            unset($tabs['additional_information']);
        }
        return $tabs;
    }

    /**
     * Shortcode to render DataAlign specs table: [dataalign_specs id="123"]. Defaults to current product.
     */
    public static function shortcode_specs($atts): string
    {
        $atts = shortcode_atts(['id' => 0], $atts, 'dataalign_specs');
        $product_id = (int) $atts['id'];
        if ($product_id <= 0 && function_exists('is_product') && is_product()) {
            $product_id = get_the_ID();
        }
        if ($product_id <= 0) {
            return '';
        }
        $attrs = \get_post_meta($product_id, '_dataalign_attributes', true);
        if (!is_array($attrs) || empty($attrs)) {
            return '';
        }
        ob_start();
        echo '<table class="shop_attributes">';
        foreach ($attrs as $k => $v) {
            $name = (string) $k;
            $value = is_array($v) ? implode(', ', array_filter(array_map('strval', $v))) : (string) $v;
            echo '<tr><th>' . \esc_html($name) . '</th><td>' . \esc_html($value) . '</td></tr>';
        }
        echo '</table>';
        return (string) ob_get_clean();
    }

    /**
     * Filter attribute labels on the frontend Additional Information tab to humanize keys with underscores/hyphens.
     * Avoids changing global attributes (pa_*) and admin screens.
     */
    public static function filter_attribute_label($label, $name, $product): string
    {
        if (is_admin()) {
            return (string) $label;
        }
        $name = (string) $name;
        $label_str = (string) $label;
        // Skip global attributes which already have proper labels
        if (strpos($name, 'pa_') === 0) {
            return $label_str;
        }
        // Only humanize when label contains underscores or hyphens
        if (!preg_match('/[_\-]/', $label_str)) {
            return $label_str;
        }
        return self::humanize_attribute_key($label_str);
    }

    /**
     * Convert attribute/spec keys like "max_speed" or "battery-capacity" to human readable labels.
     */
    protected static function humanize_attribute_key(string $key): string
    {
        $label = trim((string) $key);
        if ($label === '') { return $label; }
        // Replace underscores/hyphens with spaces and collapse whitespace
        $label = preg_replace('/[_\-]+/', ' ', $label);
        $label = preg_replace('/\s+/', ' ', (string) $label);
        $label = trim((string) $label);
        if (function_exists('mb_convert_case')) {
            return mb_convert_case($label, MB_CASE_TITLE, 'UTF-8');
        }
        // Fallback
        return ucwords(strtolower($label));
    }

    /**
     * Register meta box on WooCommerce product edit screen.
     */
    public static function register_product_metabox(): void
    {
        \add_meta_box(
            'dataalign_woo_metabox',
            \esc_html__('DataAlign AI', 'ai-product-attributes-for-woocommerce'),
            [__CLASS__, 'render_product_metabox'],
            'product',
            'side',
            'high'
        );
    }

    /**
     * Render the product meta box UI.
     */
    public static function render_product_metabox(\WP_Post $post): void
    {
        $product_id = (int) $post->ID;
        $nonce = \wp_create_nonce('dataalign_enrich_' . $product_id);

        $api_key  = (string) \get_option('dataalign_api_key', '');
        $configured = ($api_key !== '');

        $last_enriched = (string) \get_post_meta($product_id, '_dataalign_last_enriched', true);
        $last_error    = (string) \get_post_meta($product_id, '_dataalign_last_error', true);

        echo '<div class="dataalign-woo-box">';
        if (!$configured) {
            echo '<p>' . \esc_html__('API not configured. Set API key in WooCommerce → DataAlign AI.', 'ai-product-attributes-for-woocommerce') . '</p>';
        }

        echo '<p><strong>' . \esc_html__('Status', 'ai-product-attributes-for-woocommerce') . ':</strong></p>';
        echo '<div id="dataalign-status">';
        if ($last_enriched !== '') {
            // Expecting a timestamp; if string, try casting
            $ts = (int) $last_enriched;
            $when = $ts > 0 ? \date_i18n(\get_option('date_format') . ' ' . \get_option('time_format'), $ts) : (string) $last_enriched;
            echo '<p>' . \esc_html__('Last enriched:', 'ai-product-attributes-for-woocommerce') . ' ' . \esc_html($when) . '</p>';
        } else {
            echo '<p>' . \esc_html__('No enrichment yet.', 'ai-product-attributes-for-woocommerce') . '</p>';
        }
        if ($last_error !== '') {
            echo '<p style="color:#b32d2e;">' . \esc_html__('Last error:', 'ai-product-attributes-for-woocommerce') . ' ' . \esc_html($last_error) . '</p>';
        }
        echo '</div>';

        echo '<p>';
        echo '<button type="button" class="button button-primary" id="dataalign-enrich-button" ' . ($configured ? '' : 'disabled') . ' title="' . ($configured ? '' : \esc_attr__('Configure DataAlign API key first.', 'ai-product-attributes-for-woocommerce')) . '">' . \esc_html__('Enrich with DataAlign AI', 'ai-product-attributes-for-woocommerce') . '</button> ';
        echo '<span class="spinner" id="dataalign-enrich-spinner" style="float:none;"></span>';
        if (!$configured) {
            echo '<span style="margin-left:8px;color:#646970;">' . \esc_html__('Configure DataAlign API key first.', 'ai-product-attributes-for-woocommerce') . '</span>';
        }
        echo '</p>';

        // Attributes read-only table (always render wrapper so JS can refresh it)
        echo '<hr />';
        echo '<div id="dataalign-attributes-box">';
        $attrs = \get_post_meta($product_id, '_dataalign_attributes', true);
        echo '<p><strong>' . \esc_html__('DataAlign Attributes', 'ai-product-attributes-for-woocommerce') . '</strong></p>';
        if (is_array($attrs) && !empty($attrs)) {
            self::render_attributes_table($attrs);
        } else {
            echo '<p class="description" style="margin-top:6px;">' . \esc_html__('No DataAlign attributes yet. Use “Enrich with DataAlign AI”.', 'ai-product-attributes-for-woocommerce') . '</p>';
        }
        echo '</div>';

        // Sync button removed (attributes can auto-sync via setting or be updated on next save)

        // Restore baseline UI (only if baseline exists)
        $has_baseline_tags = (array) \get_post_meta($product_id, '_dataalign_preserved_tag_ids', true);
        $has_baseline_attrs = (array) \get_post_meta($product_id, '_dataalign_preserved_attributes', true);
        $has_seo_title = (string) \get_post_meta($product_id, '_dataalign_preserved_seo_title', true);
        $has_seo_desc  = (string) \get_post_meta($product_id, '_dataalign_preserved_seo_description', true);
        $can_restore = (!empty($has_baseline_tags) || !empty($has_baseline_attrs) || $has_seo_title !== '' || $has_seo_desc !== '');
        echo '<p style="margin-top:8px">';
        echo '<button type="button" class="button" id="dataalign-restore-baseline" ' . ($can_restore ? '' : 'disabled') . '>' . \esc_html__('Restore pre‑AI Tags, Attributes & SEO', 'ai-product-attributes-for-woocommerce') . '</button> ';
        echo '<span class="spinner" id="dataalign-restore-spinner" style="float:none"></span>';
        echo '<span id="dataalign-restore-status" style="margin-left:8px"></span>';
        if (!$can_restore) {
            echo '<br/><small style="color:#646970;">' . \esc_html__('Baseline not captured yet. It is recorded automatically on the first enrichment.', 'ai-product-attributes-for-woocommerce') . '</small>';
        }
        echo '</p>';

        // Nonce and product
        echo '<input type="hidden" id="dataalign-enrich-nonce" value="' . \esc_attr($nonce) . '" />';
        echo '<input type="hidden" id="dataalign-product-id" value="' . (int) $product_id . '" />';
        // Sync nonce removed since manual sync button is not shown
        echo '<input type="hidden" id="dataalign-restore-nonce" value="' . \esc_attr(\wp_create_nonce('dataalign_restore_' . $product_id)) . '" />';

        echo '</div>';
    }

    /**
     * AJAX: Return the current DataAlign Attributes HTML for a product metabox.
     */
    public static function ajax_get_attributes(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- verifying below
        $product_id = isset($_POST['product_id']) ? (int) $_POST['product_id'] : 0;
        $nonce      = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';

        if ($product_id <= 0 || !\wp_verify_nonce($nonce, 'dataalign_enrich_' . $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('edit_product', $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }

        $attrs = \get_post_meta($product_id, '_dataalign_attributes', true);

        ob_start();
        echo '<p><strong>' . \esc_html__('DataAlign Attributes', 'ai-product-attributes-for-woocommerce') . '</strong></p>';
        if (is_array($attrs) && !empty($attrs)) {
            self::render_attributes_table($attrs);
        } else {
            echo '<p class="description" style="margin-top:6px;">' . \esc_html__('No DataAlign attributes yet. Use “Enrich with DataAlign AI”.', 'ai-product-attributes-for-woocommerce') . '</p>';
        }
        $html = (string) ob_get_clean();

        \wp_send_json_success([
            'html' => $html,
        ]);
    }

    /**
     * Return WooCommerce Attributes panel rows HTML for the product, so the default
     * Attributes tab can be refreshed without a page reload.
     */
    public static function ajax_get_wc_attributes(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- verifying below
        $product_id = isset($_POST['product_id']) ? (int) $_POST['product_id'] : 0;
        $nonce      = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';

        if ($product_id <= 0 || !\wp_verify_nonce($nonce, 'dataalign_enrich_' . $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('edit_product', $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!function_exists('wc_get_product')) {
            \wp_send_json_error(['message' => \esc_html__('WooCommerce is required.', 'ai-product-attributes-for-woocommerce')]);
        }

        $product = \wc_get_product($product_id);
        if (!$product) {
            \wp_send_json_error(['message' => \esc_html__('Product not found.', 'ai-product-attributes-for-woocommerce')]);
        }

        $attributes = $product->get_attributes('edit');
        ob_start();
        $i = -1;
        foreach ($attributes as $attribute) {
            ++$i;
            $metabox_class = [];
            if ($attribute->is_taxonomy()) {
                $metabox_class[] = 'taxonomy';
                $metabox_class[] = $attribute->get_name();
            }
            $view = defined('WC_ABSPATH') ? WC_ABSPATH . 'includes/admin/meta-boxes/views/html-product-attribute.php' : WP_PLUGIN_DIR . '/woocommerce/includes/admin/meta-boxes/views/html-product-attribute.php';
            if (file_exists($view)) {
                include $view; // phpcs:ignore WordPressVIPMinimum.Files.IncludingFile.UsingCustomConstant
            }
        }
        $html = (string) ob_get_clean();

        \wp_send_json_success(['html' => $html]);
    }

    /**
     * Return current product tag names for the tags meta box refresh.
     */
    public static function ajax_get_product_tags(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- verifying below
        $product_id = isset($_POST['product_id']) ? (int) $_POST['product_id'] : 0;
        $nonce      = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';

        if ($product_id <= 0 || !\wp_verify_nonce($nonce, 'dataalign_enrich_' . $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('edit_product', $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Insufficient permissions.', 'ai-product-attributes-for-woocommerce')]);
        }

        $names = \wp_get_post_terms($product_id, 'product_tag', ['fields' => 'names']);
        if (!is_array($names)) { $names = []; }
        $names = array_values(array_filter(array_map('trim', array_map('strval', $names))));
        \wp_send_json_success(['tags' => $names]);
    }

    /**
     * Enqueue admin assets for product edit screen.
     */
    public static function enqueue_admin_assets(string $hook_suffix): void
    {
        $screen = function_exists('get_current_screen') ? \get_current_screen() : null;
        if (!$screen) {
            return;
        }

        // Settings page assets
        if ($screen->id === 'woocommerce_page_dataalign-woo') {
            \wp_enqueue_style(
                'dataalign-woo-admin',
                plugins_url('assets/css/admin.css', PLUGIN_FILE),
                [],
                VERSION
            );

            \wp_enqueue_style(
                'dataalign-woo-animated-buttons',
                plugins_url('assets/css/animated-buttons.css', PLUGIN_FILE),
                [],
                VERSION
            );

            \wp_enqueue_script(
                'dataalign-woo-settings',
                plugins_url('assets/js/settings.js', PLUGIN_FILE),
                ['jquery'],
                VERSION,
                true
            );
            \wp_enqueue_script(
                'dataalign-woo-animated-buttons',
                plugins_url('assets/js/animated-buttons.js', PLUGIN_FILE),
                [],
                VERSION,
                true
            );
            \wp_localize_script('dataalign-woo-settings', 'DataAlignWooSettings', [
                'ajaxUrl' => \admin_url('admin-ajax.php'),
                'nonce'   => \wp_create_nonce('dataalign_settings'),
                'usageEnabled' => ((string) \get_option('dataalign_api_key', '') !== ''),
                'i18n'    => [
                    'testing' => \esc_html__('Testing connection…', 'ai-product-attributes-for-woocommerce'),
                    'ok'      => \esc_html__('Connection OK', 'ai-product-attributes-for-woocommerce'),
                    'failed'  => \esc_html__('Connection failed', 'ai-product-attributes-for-woocommerce'),
                    'show'    => \esc_html__('Show', 'ai-product-attributes-for-woocommerce'),
                    'hide'    => \esc_html__('Hide', 'ai-product-attributes-for-woocommerce'),
                    'registering' => \esc_html__('Registering site…', 'ai-product-attributes-for-woocommerce'),
                    'registered'  => \esc_html__('Site registered. API key saved.', 'ai-product-attributes-for-woocommerce'),
                    'loadingUsage' => \esc_html__('Loading usage…', 'ai-product-attributes-for-woocommerce'),
                    'usageError'   => \esc_html__('Failed to load usage.', 'ai-product-attributes-for-woocommerce'),
                    'restoringAll' => \esc_html__('Restoring products to their original form…', 'ai-product-attributes-for-woocommerce'),
                    'restoreDone'  => \esc_html__('Restore complete.', 'ai-product-attributes-for-woocommerce'),
                    'restoreQueued' => \esc_html__('Queued…', 'ai-product-attributes-for-woocommerce'),
                    'restoreRunning' => \esc_html__('Running…', 'ai-product-attributes-for-woocommerce'),
                    'bulkStarting' => \esc_html__('Starting bulk optimization…', 'ai-product-attributes-for-woocommerce'),
                    'bulkQueued'   => \esc_html__('Queued…', 'ai-product-attributes-for-woocommerce'),
                    'bulkRunning'  => \esc_html__('Running…', 'ai-product-attributes-for-woocommerce'),
                    'bulkDone'     => \esc_html__('Bulk optimization complete.', 'ai-product-attributes-for-woocommerce'),
                    'bulkCancelling' => \esc_html__('Cancelling…', 'ai-product-attributes-for-woocommerce'),
                    'bulkCancelled'  => \esc_html__('Cancelled.', 'ai-product-attributes-for-woocommerce'),
                    'planLimitReached' => \esc_html__('Plan limit reached. Upgrade to continue.', 'ai-product-attributes-for-woocommerce'),
                ],
            ]);
        }

        // Product editor assets
        if ($screen->id === 'product') {
            \wp_enqueue_style(
                'dataalign-woo-animated-buttons',
                plugins_url('assets/css/animated-buttons.css', PLUGIN_FILE),
                [],
                VERSION
            );
            \wp_enqueue_script(
                'dataalign-woo-product',
                plugins_url('assets/js/product-enrich.js', PLUGIN_FILE),
                ['jquery'],
                VERSION,
                true
            );
            \wp_enqueue_script(
                'dataalign-woo-animated-buttons',
                plugins_url('assets/js/animated-buttons.js', PLUGIN_FILE),
                [],
                VERSION,
                true
            );

            $product_id = isset($_GET['post']) ? (int) $_GET['post'] : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $nonce = $product_id ? \wp_create_nonce('dataalign_enrich_' . $product_id) : '';

            \wp_localize_script('dataalign-woo-product', 'DataAlignWoo', [
                'ajaxUrl'   => \admin_url('admin-ajax.php'),
                'productId' => $product_id,
                'nonce'     => $nonce,
                'i18n'      => [
                    'working' => \esc_html__('Enriching…', 'ai-product-attributes-for-woocommerce'),
                    'success' => \esc_html__('Enrichment complete.', 'ai-product-attributes-for-woocommerce'),
                    'error'   => \esc_html__('Enrichment failed.', 'ai-product-attributes-for-woocommerce'),
                ],
            ]);
        }
    }

    /**
     * Enqueue frontend assets for product pages (WooCommerce).
     */
    public static function enqueue_frontend_assets(): void
    {
        // No frontend assets are enqueued to avoid impacting theme styles.
        // If needed in the future, enqueue only styles scoped to plugin-rendered wrappers.
        return;
    }

    /**
     * AJAX handler for product enrichment.
     */
    public static function ajax_enrich_product(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- we manually verify below
        $product_id = isset($_POST['product_id']) ? (int) $_POST['product_id'] : 0;
        $nonce      = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash((string) $_POST['nonce']) ) : '';

        if ($product_id <= 0) {
            \wp_send_json_error(['message' => \esc_html__('Invalid product ID.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\wp_verify_nonce($nonce, 'dataalign_enrich_' . $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('Security check failed.', 'ai-product-attributes-for-woocommerce')]);
        }
        if (!\current_user_can('edit_product', $product_id)) {
            \wp_send_json_error(['message' => \esc_html__('You do not have permission to edit this product.', 'ai-product-attributes-for-woocommerce')]);
        }

        if (!function_exists('wc_get_product')) {
            \wp_send_json_error(['message' => \esc_html__('WooCommerce is required.', 'ai-product-attributes-for-woocommerce')]);
        }

        $product = \wc_get_product($product_id);
        if (!$product) {
            \wp_send_json_error(['message' => \esc_html__('Product not found.', 'ai-product-attributes-for-woocommerce')]);
        }

        $post = \get_post($product_id);
        if (!$post) {
            \wp_send_json_error(['message' => \esc_html__('Product post not found.', 'ai-product-attributes-for-woocommerce')]);
        }

        // Collect payload
        $title = (string) $post->post_title;
        $description = (string) $post->post_content;

        $categories = \wp_get_post_terms($product_id, 'product_cat', ['fields' => 'names']);
        if (!is_array($categories)) {
            $categories = [];
        }

        $exclude_images = (bool) \get_option('dataalign_exclude_images', false);
        $images = [];
        if (!$exclude_images) {
            // Parent product thumbnail and gallery
            $thumb_id = (int) \get_post_thumbnail_id($product_id);
            if ($thumb_id) {
                $url = \wp_get_attachment_image_url($thumb_id, 'full');
                if ($url) { $images[] = $url; }
            }
            $gallery_ids = method_exists($product, 'get_gallery_image_ids') ? (array) $product->get_gallery_image_ids() : [];
            foreach ($gallery_ids as $img_id) {
                $url = \wp_get_attachment_image_url((int) $img_id, 'full');
                if ($url) { $images[] = $url; }
            }
            // Include variation images for variable products
            if (method_exists($product, 'is_type') && $product->is_type('variable')) {
                $variation_ids = method_exists($product, 'get_children') ? (array) $product->get_children() : [];
                foreach ($variation_ids as $vid) {
                    $v = function_exists('wc_get_product') ? \wc_get_product((int) $vid) : null;
                    if ($v && method_exists($v, 'get_image_id')) {
                        $v_img_id = (int) $v->get_image_id();
                        if ($v_img_id) {
                            $v_url = \wp_get_attachment_image_url($v_img_id, 'full');
                            if ($v_url) { $images[] = $v_url; }
                        }
                    }
                }
            }
            // Rewrite to public base if provided
            $public_base = (string) \get_option('dataalign_public_base_url', '');
            $site_base   = \home_url();
            if ($public_base !== '' && $site_base) {
                $public_base = rtrim($public_base, '/');
                $site_base   = rtrim($site_base, '/');
                $images = array_map(function ($u) use ($public_base, $site_base) {
                    return is_string($u) ? str_replace($site_base, $public_base, $u) : $u;
                }, $images);
            }
            // De-duplicate
            $images = array_values(array_unique(array_filter(array_map('strval', $images))));
        }

        // Detect SEO plugin + limits for this product
        $seo_plugin = self::detect_seo_plugin($product_id);
        $seo_limits = self::get_seo_limits($seo_plugin['slug'] ?? 'default');
        // Gather existing store tags to help API reuse terms
        $existing_tags = self::get_all_product_tag_names();

        $payload = [
            'product_id'  => (string) $product_id,
            'title'       => $title,
            'description' => $description,
            'categories'  => array_values(array_map('strval', $categories)),
            'images'      => array_values(array_map('strval', $images)),
            'existing_tags' => $existing_tags,
            'seo_plugin'       => (string) ($seo_plugin['slug'] ?? 'none'),
            'seo_plugin_name'  => (string) ($seo_plugin['name'] ?? 'None'),
            'seo_limits'       => [
                'title_max'       => (int) ($seo_limits['title_max'] ?? 60),
                'description_max' => (int) ($seo_limits['description_max'] ?? 160),
            ],
        ];

        $client = new DataAlign_API_Client();
        $result = $client->enrich_product($payload);

        if (\is_wp_error($result)) {
            \update_post_meta($product_id, '_dataalign_last_error', $result->get_error_message());
            $err = [ 'message' => \esc_html($result->get_error_message()) ];
            if ($result->get_error_code() === 'dataalign_plan_limit') {
                $err['upgradeUrl'] = defined(__NAMESPACE__ . '\\UPGRADE_URL') ? constant(__NAMESPACE__ . '\\UPGRADE_URL') : 'https://dataalign.ai/pricing';
            }
            \wp_send_json_error($err);
        }

        // Expecting structure: success (bool), data{ tags[], attributes{}, seo{title, meta_description}, job_id }
        $success = isset($result['success']) ? (bool) $result['success'] : false;
        $data    = isset($result['data']) && is_array($result['data']) ? $result['data'] : [];

        if (!$success || empty($data)) {
            $msg = isset($result['message']) ? (string) $result['message'] : __('API returned no data.', 'ai-product-attributes-for-woocommerce');
            \update_post_meta($product_id, '_dataalign_last_error', $msg);
            \wp_send_json_error(['message' => \esc_html($msg)]);
        }

        // Apply mapping to product
        self::apply_enrichment_data($product_id, $data, $result);

        \wp_send_json_success([
            'message' => \esc_html__('Product enriched successfully.', 'ai-product-attributes-for-woocommerce'),
            'tags'    => isset($data['tags']) && is_array($data['tags']) ? $data['tags'] : [],
            'seo'     => isset($data['seo']) && is_array($data['seo']) ? $data['seo'] : [],
        ]);
    }

    /**
     * Add custom bulk action to products list table.
     */
    public static function register_bulk_actions(array $bulk_actions): array
    {
        $api_key  = (string) \get_option('dataalign_api_key', '');
        $configured = ($api_key !== '');

        if ($configured) {
        $bulk_actions['dataalign_enrich_bulk'] = \esc_html__('Enrich with DataAlign AI (Bulk)', 'ai-product-attributes-for-woocommerce');
        }
        return $bulk_actions;
    }

    /**
     * Handle our bulk action for selected products.
     */
    public static function handle_bulk_actions(string $redirect_url, string $action, array $post_ids): string
    {
        if ($action !== 'dataalign_enrich_bulk' || empty($post_ids)) {
            return $redirect_url;
        }
        // Filter selected IDs to valid products
        $targets = [];
        foreach ($post_ids as $pid) {
            $pid = (int) $pid;
            $post = \get_post($pid);
            if ($post && $post->post_type === 'product') {
                $targets[] = $pid;
            }
        }
        $targets = array_values(array_unique(array_map('intval', $targets)));
        if (empty($targets)) {
            return $redirect_url;
        }

        // Seed a targeted bulk job and process first tick inline for responsiveness
        $total = count($targets);
        $per_page = 25;
        $run_id = uniqid('da_run_', true);
        $state = [
            'status'     => 'queued',
            'total'      => $total,
            'processed'  => 0,
            'applied'    => 0,
            'per_page'   => $per_page,
            'page'       => 1,
            'run_id'     => $run_id,
            'targets'    => $targets,
            'started_at' => time(),
            'updated_at' => time(),
        ];
        \update_option('_dataalign_bulk_enrich_job', $state, false);
        // Legacy meta-based queuing removed; redirect to Optimize tab for the table-driven pipeline

        // Redirect to Optimize tab for live progress
        $settings_url = \admin_url('admin.php?page=dataalign-woo');
        // Add fragment to switch to Optimize tab on load
        $settings_url .= '#optimize';
        return $settings_url;
    }

    /**
     * Register custom cron schedules.
     */
    public static function register_cron_schedules(array $schedules): array
    {
        if (!isset($schedules['dataalign_every_minute'])) {
            $schedules['dataalign_every_minute'] = [
                'interval' => 60,
                'display'  => \esc_html__('Every Minute (DataAlign)', 'ai-product-attributes-for-woocommerce'),
            ];
        }
        if (!isset($schedules['dataalign_every_five_minutes'])) {
            $schedules['dataalign_every_five_minutes'] = [
                'interval' => 5 * 60,
                'display'  => \esc_html__('Every Five Minutes (DataAlign)', 'ai-product-attributes-for-woocommerce'),
            ];
        }
        return $schedules;
    }

    /**
     * Apply pending job results for products with queued jobs.
     * When called from cron, uses conservative limits; when called from AJAX, can use higher limits.
     */
    protected static function apply_pending_jobs(int $max_products = 10, int $max_jobs_per_product = 5): void
    {
        $q = new \WP_Query([
            'post_type'      => 'product',
            'post_status'    => 'any',
            'posts_per_page' => max(1, $max_products),
            'fields'         => 'ids',
            'no_found_rows'  => true,
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Select products flagged with pending jobs
            'meta_query'     => [
                [
                    'key'   => '_dataalign_has_pending_jobs',
                    'value' => true,
                ],
            ],
        ]);

        if (empty($q->posts)) {
            // No pending products — clear global pending flag so cron can short-circuit next pass
            \update_option('_dataalign_any_pending', 0, false);
            return;
        }

        $client = new DataAlign_API_Client();

        // Attempt a bulk status request first (if backend supports /api/v1/jobs/status)
        $map_job_to_product = [];
        $bulk_ids = [];
        foreach ($q->posts as $pid_pre) {
            $pid_pre = (int) $pid_pre;
            $all_pre = (array) \get_post_meta($pid_pre, '_dataalign_job_ids', true);
            $done_pre = (array) \get_post_meta($pid_pre, '_dataalign_completed_job_ids', true);
            $pending_pre = array_values(array_diff(array_map('intval', $all_pre), array_map('intval', $done_pre)));
            if (empty($pending_pre)) { continue; }
            $pending_pre = array_slice($pending_pre, 0, max(1, $max_jobs_per_product));
            foreach ($pending_pre as $jid) { $map_job_to_product[(int)$jid] = $pid_pre; $bulk_ids[] = (int) $jid; }
        }

        $bulk_used = false;
        if (!empty($bulk_ids)) {
            $bulk_ids = array_values(array_unique($bulk_ids));
            $multi = $client->get_jobs_status($bulk_ids);
            if (!\is_wp_error($multi) && isset($multi['data']['jobs']) && is_array($multi['data']['jobs'])) {
                foreach ($multi['data']['jobs'] as $j) {
                    $jid = isset($j['id']) ? (int) $j['id'] : 0;
                    if ($jid <= 0 || !isset($map_job_to_product[$jid])) { continue; }
                    $pid_m = (int) $map_job_to_product[$jid];
                    $status = isset($j['status']) ? (string) $j['status'] : '';
                    $result = isset($j['result']) && is_array($j['result']) ? $j['result'] : null;
                    if ($status === 'completed' && !empty($result)) {
                        self::apply_enrichment_data($pid_m, $result, is_array($j) ? $j : []);
                        $bulk_state = \get_option('_dataalign_bulk_enrich_job', []);
                        if (is_array($bulk_state) && !empty($bulk_state) && isset($bulk_state['run_id'])) {
                            $run_id = (string) $bulk_state['run_id'];
                            $b_status = isset($bulk_state['status']) ? (string) $bulk_state['status'] : 'queued';
                            if ($run_id !== '' && $b_status !== 'cancelled') {
                                $p_run = (string) \get_post_meta($pid_m, '_dataalign_bulk_run', true);
                                $p_applied = (string) \get_post_meta($pid_m, '_dataalign_bulk_applied', true);
                                if ($p_run === $run_id && $p_applied !== $run_id) {
                                    $applied = isset($bulk_state['applied']) ? (int) $bulk_state['applied'] : 0;
                                    $bulk_state['applied'] = (int) ($applied + 1);
                                    $bulk_state['updated_at'] = time();
                                    $total = isset($bulk_state['total']) ? (int) $bulk_state['total'] : 0;
                                    if ($total > 0 && $bulk_state['applied'] >= $total) { $bulk_state['status'] = 'completed'; }
                                    \update_option('_dataalign_bulk_enrich_job', $bulk_state, false);
                                    \update_post_meta($pid_m, '_dataalign_bulk_applied', $run_id);
                                }
                            }
                        }
                        $completed_jobs_m = (array) \get_post_meta($pid_m, '_dataalign_completed_job_ids', true);
                        $completed_jobs_m[] = $jid;
                        $completed_jobs_m = array_values(array_unique(array_map('intval', $completed_jobs_m)));
                        \update_post_meta($pid_m, '_dataalign_completed_job_ids', $completed_jobs_m);
                    }
                }
                // Clean per-product pending flags if done
                foreach ($q->posts as $pid_chk) {
                    $pid_chk = (int) $pid_chk;
                    $all_chk = (array) \get_post_meta($pid_chk, '_dataalign_job_ids', true);
                    $done_chk = (array) \get_post_meta($pid_chk, '_dataalign_completed_job_ids', true);
                    $pending_chk = array_values(array_diff(array_map('intval', $all_chk), array_map('intval', $done_chk)));
                    if (empty($pending_chk)) { \update_post_meta($pid_chk, '_dataalign_has_pending_jobs', false); }
                }
                $bulk_used = true;
            }
        }

        if ($bulk_used) {
            return; // Bulk path handled this pass
        }

        foreach ($q->posts as $pid) {
            $pid = (int) $pid;
            $all_jobs = (array) \get_post_meta($pid, '_dataalign_job_ids', true);
            $completed_jobs = (array) \get_post_meta($pid, '_dataalign_completed_job_ids', true);
            $pending = array_values(array_diff(array_map('intval', $all_jobs), array_map('intval', $completed_jobs)));
            if (empty($pending)) {
                \update_post_meta($pid, '_dataalign_has_pending_jobs', false);
                continue;
            }
            $pending = array_slice($pending, 0, max(1, $max_jobs_per_product));
            foreach ($pending as $job_id) {
                $resp = $client->get_job((int) $job_id);
                if (\is_wp_error($resp)) {
                    \update_post_meta($pid, '_dataalign_last_error', $resp->get_error_message());
                    \update_option('_dataalign_last_global_error', $resp->get_error_message());
                    continue;
                }
                $success = isset($resp['success']) ? (bool) $resp['success'] : false;
                if (!$success) { continue; }
                $data = isset($resp['data']) && is_array($resp['data']) ? $resp['data'] : [];
                $status = isset($data['status']) ? (string) $data['status'] : '';
                $result = isset($data['result']) && is_array($data['result']) ? $data['result'] : null;
                if ($status === 'completed' && !empty($result)) {
                    self::apply_enrichment_data($pid, $result, $resp);
                    $bulk_state = \get_option('_dataalign_bulk_enrich_job', []);
                    if (is_array($bulk_state) && !empty($bulk_state) && isset($bulk_state['run_id'])) {
                        $run_id = (string) $bulk_state['run_id'];
                        $b_status = isset($bulk_state['status']) ? (string) $bulk_state['status'] : 'queued';
                        if ($run_id !== '' && $b_status !== 'cancelled') {
                            $p_run = (string) \get_post_meta($pid, '_dataalign_bulk_run', true);
                            $p_applied = (string) \get_post_meta($pid, '_dataalign_bulk_applied', true);
                            if ($p_run === $run_id && $p_applied !== $run_id) {
                                $applied = isset($bulk_state['applied']) ? (int) $bulk_state['applied'] : 0;
                                $bulk_state['applied'] = (int) ($applied + 1);
                                $bulk_state['updated_at'] = time();
                                $total = isset($bulk_state['total']) ? (int) $bulk_state['total'] : 0;
                                if ($total > 0 && $bulk_state['applied'] >= $total) {
                                    $bulk_state['status'] = 'completed';
                                }
                                \update_option('_dataalign_bulk_enrich_job', $bulk_state, false);
                                \update_post_meta($pid, '_dataalign_bulk_applied', $run_id);
                            }
                        }
                    }
                    $completed_jobs[] = (int) $job_id;
                    $completed_jobs = array_values(array_unique(array_map('intval', $completed_jobs)));
                    \update_post_meta($pid, '_dataalign_completed_job_ids', $completed_jobs);
                }
            }
            // Recompute pending
            $all_jobs = (array) \get_post_meta($pid, '_dataalign_job_ids', true);
            $completed_jobs = (array) \get_post_meta($pid, '_dataalign_completed_job_ids', true);
            $pending_after = array_values(array_diff(array_map('intval', $all_jobs), array_map('intval', $completed_jobs)));
            if (empty($pending_after)) {
                \update_post_meta($pid, '_dataalign_has_pending_jobs', false);
            }
        }
    }

    /**
     * Cron callback: conservative background apply pass.
     */
    public static function cron_check_jobs(): void
    {
        $bulk = (array) \get_option('_dataalign_bulk_enrich_job', []);
        $run_id = isset($bulk['run_id']) ? (string) $bulk['run_id'] : '';
        if ($run_id === '' || (isset($bulk['status']) && $bulk['status'] === 'cancelled')) { return; }
        global $wpdb; $table = \esc_sql(self::jobs_table());
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $queued = (int) $wpdb->get_var(
            $wpdb->prepare(
                /* phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared */
                "SELECT COUNT(*) FROM `{$table}` WHERE run_id=%s AND status=%s",
                $run_id,
                'queued'
            )
        );
        if ($queued <= 0) { return; }
        self::process_apply_from_table(50);
    }

    /**
     * Apply enrichment mapping to a product (tags, attributes, SEO, job id, timestamps).
     *
     * @param int   $product_id
     * @param array $data       Expected fields: tags[], attributes{}, seo{title, meta_description}, job_id
     * @param array $raw_result Raw API payload (optional for debugging)
     */
    protected static function apply_enrichment_data(int $product_id, array $data, array $raw_result = []): void
    {
        // Tags
        $tags = isset($data['tags']) && is_array($data['tags']) ? array_map('strval', $data['tags']) : [];
        // Always record baseline tags (the tags that existed before the first enrichment) so we never remove them unintentionally
        $baseline_ids = \get_post_meta($product_id, '_dataalign_preserved_tag_ids', true);
        if (!is_array($baseline_ids)) { $baseline_ids = []; }
        if (empty($baseline_ids)) {
            $existing_ids = \wp_get_post_terms($product_id, 'product_tag', ['fields' => 'ids']);
            if (\is_wp_error($existing_ids) || !is_array($existing_ids)) { $existing_ids = []; }
            $existing_ids = array_values(array_unique(array_map('intval', $existing_ids)));
            \update_post_meta($product_id, '_dataalign_preserved_tag_ids', $existing_ids);
            $baseline_ids = $existing_ids;
        }
        // Only attempt to apply AI tags when provided
        if (!empty($tags)) {
            // Resolve AI tag names to existing terms when possible to avoid duplicates
            $ai_ids = [];
            foreach ($tags as $name) {
                $tid = self::resolve_product_tag_id((string) $name);
                if ($tid > 0) { $ai_ids[] = $tid; }
            }
            $ai_ids = array_values(array_unique(array_filter(array_map('intval', $ai_ids))));

            $override = (bool) \get_option('dataalign_override_existing_tags', false);
            // Preserve baseline; replace previous AI-only tags with the current AI set to avoid unbounded growth
            $final_ids = $override ? $ai_ids : array_values(array_unique(array_merge($baseline_ids, $ai_ids)));
            if (!empty($final_ids)) {
                \wp_set_object_terms($product_id, $final_ids, 'product_tag', false);
                // Track last AI-applied tag IDs for reference
                \update_post_meta($product_id, '_dataalign_ai_tag_ids', $ai_ids);
            }
        }

        // Attributes/specs (baseline-preserving)
        $incoming_attrs = isset($data['attributes']) && is_array($data['attributes']) ? $data['attributes'] : [];
        // Normalize incoming to string values (implode arrays)
        $normalized_incoming = [];
        foreach ($incoming_attrs as $k => $v) {
            $name = trim((string) $k);
            if ($name === '') { continue; }
            $val = is_array($v) ? implode(', ', array_filter(array_map('strval', $v))) : (string) $v;
            $val = trim($val);
            if ($val === '') { continue; }
            $normalized_incoming[$name] = $val;
        }
        $override_attrs = (bool) \get_option('dataalign_override_existing_attributes', false);
        $baseline_attrs = \get_post_meta($product_id, '_dataalign_preserved_attributes', true);
        if (!is_array($baseline_attrs)) { $baseline_attrs = []; }
        if (empty($baseline_attrs)) {
            $existing_da_attrs = \get_post_meta($product_id, '_dataalign_attributes', true);
            if (!is_array($existing_da_attrs)) { $existing_da_attrs = []; }
            // Normalize existing DataAlign attributes (if any)
            $normalized_existing = [];
            foreach ($existing_da_attrs as $ek => $ev) {
                $ename = trim((string) $ek);
                if ($ename === '') { continue; }
                $eval = is_array($ev) ? implode(', ', array_filter(array_map('strval', $ev))) : (string) $ev;
                $eval = trim($eval);
                if ($eval === '') { continue; }
                $normalized_existing[$ename] = $eval;
            }

            // If no existing DataAlign attributes, seed baseline from WooCommerce product attributes
            if (empty($normalized_existing)) {
                try {
                    $woo_seed = [];
                    if (function_exists('wc_get_product')) {
                        $prod = \wc_get_product($product_id);
                        if ($prod && method_exists($prod, 'get_attributes')) {
                            $atts = (array) $prod->get_attributes();
                            foreach ($atts as $att) {
                                // Newer Woo returns WC_Product_Attribute objects
                                if (is_object($att) && method_exists($att, 'get_name')) {
                                    $raw_name = (string) $att->get_name();
                                    $label = function_exists('wc_attribute_label') ? (string) \wc_attribute_label($raw_name) : $raw_name;
                                    $values = [];
                                    $options = (array) $att->get_options();
                                    if (method_exists($att, 'is_taxonomy') && $att->is_taxonomy()) {
                                        foreach ($options as $tid) {
                                            $term = \get_term((int) $tid, $raw_name);
                                            if ($term && !\is_wp_error($term)) {
                                                $values[] = (string) $term->name;
                                            }
                                        }
                                    } else {
                                        $values = array_map('strval', $options);
                                    }
                                    $val = trim(implode(', ', array_filter(array_map('trim', $values))));
                                    $label = trim($label);
                                    if ($label !== '' && $val !== '') {
                                        $woo_seed[$label] = $val;
                                    }
                                // Fallback for array-shaped attribute meta
                                } elseif (is_array($att)) {
                                    $raw_name = isset($att['name']) ? (string) $att['name'] : '';
                                    if ($raw_name === '') { continue; }
                                    $label = function_exists('wc_attribute_label') ? (string) \wc_attribute_label($raw_name) : $raw_name;
                                    $val = '';
                                    $is_tax = !empty($att['is_taxonomy']);
                                    if ($is_tax) {
                                        $terms = \wp_get_post_terms($product_id, $raw_name, ['fields' => 'names']);
                                        if (is_array($terms)) {
                                            $val = trim(implode(', ', array_filter(array_map('strval', $terms))));
                                        }
                                    } else {
                                        $raw_val = isset($att['value']) ? (string) $att['value'] : '';
                                        // Custom attributes often stored as pipe-delimited
                                        $parts = preg_split('/\s*\|\s*/', $raw_val);
                                        if (is_array($parts)) {
                                            $val = trim(implode(', ', array_filter(array_map('trim', $parts))));
                                        }
                                    }
                                    $label = trim($label);
                                    if ($label !== '' && $val !== '') {
                                        $woo_seed[$label] = $val;
                                    }
                                }
                            }
                        }
                    }
                    if (!empty($woo_seed)) {
                        $normalized_existing = $woo_seed;
                    }
                } catch (\Throwable $e) {
                    // Silently ignore seeding errors; fall back to empty baseline
                }
            }

            \update_post_meta($product_id, '_dataalign_preserved_attributes', $normalized_existing);
            $baseline_attrs = $normalized_existing;
            // Also snapshot Woo custom attributes for exact restore
            $wc_attrs = \get_post_meta($product_id, '_product_attributes', true);
            if (is_array($wc_attrs)) {
                \update_post_meta($product_id, '_dataalign_preserved_wc_attributes', $wc_attrs);
            }
        }
        if (!empty($normalized_incoming)) {
            if ($override_attrs) {
                \update_post_meta($product_id, '_dataalign_attributes', $normalized_incoming);
            } else {
                // Preserve baseline values; add new AI-only keys
                $merged = $baseline_attrs;
                foreach ($normalized_incoming as $nk => $nv) {
                    if (!array_key_exists($nk, $merged)) {
                        $merged[$nk] = $nv;
                    }
                }
                \update_post_meta($product_id, '_dataalign_attributes', $merged);
            }
        }

        // SEO
        $seo = isset($data['seo']) && is_array($data['seo']) ? $data['seo'] : [];
        $seo_title = isset($seo['title']) ? (string) $seo['title'] : '';
        $seo_desc  = isset($seo['meta_description']) ? (string) $seo['meta_description'] : '';
        // Preserve baseline SEO (title/description) once before we write anything
        $pres_t = (string) \get_post_meta($product_id, '_dataalign_preserved_seo_title', true);
        $pres_d = (string) \get_post_meta($product_id, '_dataalign_preserved_seo_description', true);
        if ($pres_t === '' && $pres_d === '') {
            $existing_seo = self::get_current_seo($product_id);
            if (($existing_seo['title'] ?? '') !== '' || ($existing_seo['meta_description'] ?? '') !== '') {
                \update_post_meta($product_id, '_dataalign_preserved_seo_title', (string) ($existing_seo['title'] ?? ''));
                \update_post_meta($product_id, '_dataalign_preserved_seo_description', (string) ($existing_seo['meta_description'] ?? ''));
                $det = self::detect_seo_plugin($product_id);
                \update_post_meta($product_id, '_dataalign_preserved_seo_plugin', (string) ($det['slug'] ?? 'none'));
            }
        }
        if ($seo_title !== '') {
            \update_post_meta($product_id, '_dataalign_seo_title', $seo_title);
        }
        if ($seo_desc !== '') {
            \update_post_meta($product_id, '_dataalign_seo_meta_description', $seo_desc);
        }

        // Optionally push to popular SEO plugins
        $map_seo = (bool) \get_option('dataalign_map_seo_to_plugins', false);
        if ($map_seo && ($seo_title !== '' || $seo_desc !== '')) {
            $fill_only  = (bool) \get_option('dataalign_seo_fill_only_empty', true);
            $seo_plugin = self::detect_seo_plugin($product_id);
            $limits     = self::get_seo_limits($seo_plugin['slug'] ?? 'default');
            self::map_seo_to_plugins($product_id, $seo_title, $seo_desc, $fill_only, (string) ($seo_plugin['slug'] ?? 'none'), $limits);
        }

        if (isset($data['job_id'])) {
            \update_post_meta($product_id, '_dataalign_last_job_id', (int) $data['job_id']);
        }

        // Debugging data
        if (!empty($raw_result)) {
            \update_post_meta($product_id, '_dataalign_last_result', $raw_result);
        }

        // Timestamps and clear error
        \update_post_meta($product_id, '_dataalign_last_enriched', time());
        \delete_post_meta($product_id, '_dataalign_last_error');

        // Optional auto-sync to Woo attributes
        $auto = (bool) \get_option('dataalign_sync_attributes_to_woo', false);
        if ($auto && !empty($incoming_attrs)) {
            $fill_only = ! (bool) \get_option('dataalign_override_existing_attributes', false);
            // Use what we stored (either merged or overridden)
            $to_sync = \get_post_meta($product_id, '_dataalign_attributes', true);
            if (!is_array($to_sync)) { $to_sync = []; }
            self::sync_attributes_to_woo($product_id, $to_sync, $fill_only);
        }
    }

    /**
     * Map SEO title/description into well-known SEO plugins when present.
     * Plugins: Yoast SEO, Rank Math, SEOPress, All in One SEO (AIOSEO), The SEO Framework (TSF).
     */
    protected static function map_seo_to_plugins(int $post_id, string $title, string $desc, bool $only_fill = true, string $plugin_slug = 'all', array $limits = []): void
    {
        $title = trim($title);
        $desc  = trim($desc);
        $title_max = isset($limits['title_max']) ? (int) $limits['title_max'] : 0;
        $desc_max  = isset($limits['description_max']) ? (int) $limits['description_max'] : 0;
        if ($title_max > 0) { $title = self::truncate_to_limit($title, $title_max); }
        if ($desc_max > 0) { $desc  = self::truncate_to_limit($desc,  $desc_max); }

        // Yoast SEO
        if (($plugin_slug === 'yoast' || $plugin_slug === 'all') && (defined('WPSEO_VERSION') || class_exists('WPSEO_Meta'))) {
            $yoast_title_key = '_yoast_wpseo_title';
            $yoast_desc_key  = '_yoast_wpseo_metadesc';
            if ($title !== '') {
                $current = (string) \get_post_meta($post_id, $yoast_title_key, true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, $yoast_title_key, $title);
                }
            }
            if ($desc !== '') {
                $current = (string) \get_post_meta($post_id, $yoast_desc_key, true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, $yoast_desc_key, $desc);
                }
            }
        }

        // Rank Math
        if (($plugin_slug === 'rank_math' || $plugin_slug === 'all') && (defined('RANK_MATH_VERSION') || class_exists('RankMath\\Helper') || function_exists('rank_math'))) {
            $rm_title_key = 'rank_math_title';
            $rm_desc_key  = 'rank_math_description';
            if ($title !== '') {
                $current = (string) \get_post_meta($post_id, $rm_title_key, true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, $rm_title_key, $title);
                }
            }
            if ($desc !== '') {
                $current = (string) \get_post_meta($post_id, $rm_desc_key, true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, $rm_desc_key, $desc);
                }
            }
        }

        // SEOPress
        if (($plugin_slug === 'seopress' || $plugin_slug === 'all') && (defined('SEOPRESS_VERSION') || function_exists('seopress_init'))) {
            $sp_title_key = '_seopress_titles_title';
            $sp_desc_key  = '_seopress_titles_desc';
            if ($title !== '') {
                $current = (string) \get_post_meta($post_id, $sp_title_key, true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, $sp_title_key, $title);
                }
            }
            if ($desc !== '') {
                $current = (string) \get_post_meta($post_id, $sp_desc_key, true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, $sp_desc_key, $desc);
                }
            }
        }

        // All in One SEO (AIOSEO)
        if (($plugin_slug === 'aioseo' || $plugin_slug === 'all') && (defined('AIOSEO_VERSION') || class_exists('AIOSEO\\Plugin'))) {
            // Newer AIOSEO v4 stores settings in a single array meta
            $aioseo = \get_post_meta($post_id, '_aioseo', true);
            if (!is_array($aioseo)) { $aioseo = []; }
            $changed = false;
            if ($title !== '') {
                $current = isset($aioseo['title']) ? (string) $aioseo['title'] : '';
                if (!$only_fill || $current === '') {
                    $aioseo['title'] = $title;
                    $changed = true;
                }
            }
            if ($desc !== '') {
                $current = isset($aioseo['description']) ? (string) $aioseo['description'] : '';
                if (!$only_fill || $current === '') {
                    $aioseo['description'] = $desc;
                    $changed = true;
                }
            }
            if ($changed) {
                \update_post_meta($post_id, '_aioseo', $aioseo);
            }
            // Back-compat (older AIOSEO Pack keys)
            if ($title !== '') {
                $current = (string) \get_post_meta($post_id, '_aioseop_title', true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, '_aioseop_title', $title);
                }
            }
            if ($desc !== '') {
                $current = (string) \get_post_meta($post_id, '_aioseop_description', true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, '_aioseop_description', $desc);
                }
            }
        }

        // The SEO Framework (TSF)
        if (($plugin_slug === 'tsf' || $plugin_slug === 'all') && (defined('THE_SEO_FRAMEWORK_VERSION') || class_exists('The_SEO_Framework'))) {
            $tsf_title_key = '_genesis_title';
            $tsf_desc_key  = '_genesis_description';
            if ($title !== '') {
                $current = (string) \get_post_meta($post_id, $tsf_title_key, true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, $tsf_title_key, $title);
                }
            }
            if ($desc !== '') {
                $current = (string) \get_post_meta($post_id, $tsf_desc_key, true);
                if (!$only_fill || $current === '') {
                    \update_post_meta($post_id, $tsf_desc_key, $desc);
                }
            }
        }
    }

    /** Ensure the custom jobs table exists */
    public static function maybe_create_jobs_table(): void
    {
        global $wpdb; $table = $wpdb->prefix . 'dataalign_jobs';
        $charset = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE {$table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            run_id varchar(64) NOT NULL,
            product_id bigint(20) unsigned NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'ready',
            job_id bigint(20) unsigned DEFAULT NULL,
            error text NULL,
            queued_at datetime NULL,
            applied_at datetime NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            UNIQUE KEY run_product (run_id, product_id),
            KEY status (status),
            KEY job_id (job_id),
            KEY product_id (product_id)
        ) {$charset};";
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }
}

/**
 * Activation hook.
 */
function activate_plugin(): void
{
    // Schedule cron event if not already scheduled
    if (!\wp_next_scheduled('dataalign_cron_check_jobs')) {
        // Ensure schedule exists (prefer every minute)
        \wp_schedule_event(time() + 60, 'dataalign_every_minute', 'dataalign_cron_check_jobs');
    }
    // Create the custom jobs table used to drive bulk optimization
    Plugin::maybe_create_jobs_table();
    // Set default: auto-sync attributes to Woo enabled on first activation
    if (false === \get_option('dataalign_sync_attributes_to_woo')) {
        \add_option('dataalign_sync_attributes_to_woo', 1);
    }
    // Ensure "Override Existing Attributes" starts unchecked for fresh installs
    if (false === \get_option('dataalign_override_existing_attributes')) {
        \add_option('dataalign_override_existing_attributes', 0);
    }
    // Ensure "Override Existing Product Tags" starts unchecked for fresh installs
    if (false === \get_option('dataalign_override_existing_tags')) {
        \add_option('dataalign_override_existing_tags', 0);
    }
}

/**
 * Deactivation hook.
 */
function deactivate_plugin(): void
{
    // Clear scheduled cron event
    \wp_clear_scheduled_hook('dataalign_cron_check_jobs');
}

\register_activation_hook(PLUGIN_FILE, __NAMESPACE__ . '\\activate_plugin');
\register_deactivation_hook(PLUGIN_FILE, __NAMESPACE__ . '\\deactivate_plugin');

// Bootstrap
Plugin::init();
